/**
 * @file   AXFObjectName.hpp
 * @brief  AXFObjectNameクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFOBJECTNAME_HPP_
#define AXFOBJECTNAME_HPP_

#include <string>

/**
 * @class AXFObjectName
 * @brief アクタ名を操作する機能を提供する
 *
 */
class AXFObjectName {
 private:
  std::string m_actor;
  std::string m_stage;
  std::string m_module;
  std::string m_unit;
  std::string m_node;
  std::string m_site;
  std::string m_world;
  std::string m_full;

 public:
  /**
   * @brief AXFObjectNameコンストラクタ
   * @n     アクタ名を設定する
   * @n
   * @param [in] actor  Thread または Task の名前
   * @param [in] stage  複数のアクタをまとめて管理するコンテナの名前
   * @param [in] module Unitを構成するドメインの名前
   * @param [in] unit   ハードウェア種別を管理するドメインの名前
   * @param [in] node   複数の機器が密結合されたコンピュータシステム単位を示すドメインの名前
   * @param [in] site   複数の Node を収容し、特定のエリアや特定のサービスを提供するドメインの名前
   * @param [in] world  AXプラットフォームで動作する全ての環境を収容するトップドメインの名前
   */
  AXFObjectName(std::string actor, std::string stage = "", std::string module =
                    "",
                std::string unit = "", std::string node = "", std::string site =
                    "",
                std::string world = "");

  /**
   * @brief AXFObjectNameデストラクタ
   */
  virtual ~AXFObjectName();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ Stage か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentStage();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ Module か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentModule();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ Unit か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentUnit();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ Node か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentNode();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ Site か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentSite();

  /**
   * @brief  指定したアクタ名オブジェクトが、呼び出し元と同じ World か確認する
   * @return
   * -    true：呼び出し元と同じ
   * -    false：それ以外
   */
  bool isCurrentWorld();

  /**
   * @brief  フルアクタ名を文字列で取得する
   * @return "." で連結したフルアクタ名
   */
  const std::string& getFullActorName();

  /**
   * @brief  アクタ名を文字列で取得する
   */
  const std::string& getActorName();

  /**
   * @brief  Stage名を文字列で取得する
   */
  const std::string& getStageName();

  /**
   * @brief  指定されたnameと一致しているか比較する
   * @return
   * -    true：指定されたnameと一致
   * -    false：それ以外
   */
  bool isMatch(const AXFObjectName& name);

};

#endif /* AXFOBJECTNAME_HPP_ */
