/**
 * @file   AXFStateMachineInternal.hpp
 * @brief  AXFStateMachineクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFSTATEMACHINEINTERNAL_HPP_
#define AXFSTATEMACHINEINTERNAL_HPP_
#include <vector>

#include "AXFStatus.hpp"
#include "AXFEvent.hpp"
#include "AXFActor.hpp"

/**
 * @class AXFStateMachineInternal
 * @brief ライフサイクルイベント以外のイベント登録・状態制御機能を提供する
 *
 * @n      使用方法は、AXM_ISPのAXMIspを参照してください。
 */
class AXFStateMachineInternal {
 private:
  AXFActor* m_actor;
  int m_state;
  int m_stateMax;
  ax::actorFuncStatus m_stateFuncResult;

  friend class AXFActor;

 public:
  AXFStateMachineInternal();
  virtual ~AXFStateMachineInternal();


 protected:
  /**
   * @brief StateMachine に必要な情報を登録する
   *
   * @param[in] *obj          アクタのオブジェクト
   * @param[in] initState    状態遷移の初期状態
   * @param[in] maxState     状態の最大数  (状態の定義は、enumで0オリジンで作成されていること）
   * @return 作成可否
   */
  int setStateInfo(AXFActor* obj, int initState, int maxState);

  /**
   * @brief StateMachine に状態テーブルを登録する
   * @n
   * @n      第1引数で指定する状態で動作する状態テーブルを登録する
   * @n
   * @param[in] registState    登録する状態
   * @param[in] *stateTable    状態テーブルのポインタ
   */
  void setStateTable(int registState,
                     const AXFActor::StateTable* stateTable);

  /**
   * @brief StateMachine に次に遷移する状態を設定する
   *
   * @param[in] nextState    遷移先の状態
   */
  void setNextState(int nextState);

  /**
   * @brief StateMachine の実行を行う
   * @n
   * @param[in] eventId  イベントID
   * @param[in] *pParam  データのポインタ
   * @param[in] size     データのサイズ
   * @return 実行完了状態
   */
  ax::actorFuncStatus doHandler(AXFEvent_t eventId, const void *pParam,
                                int size);

  /**
   * @brief 現在の状態を返す
   * @n
   * @return 現在、設定されている状態を返す
   */
  int getCurrentState();

  /**
   * @brief StateMachine の実行結果を返す
   * @n
   * @return 実行された状態遷移の結果を返す
   */
  ax::actorFuncStatus getExecutionResult();

 private:
  /**
   * @class AXFStateTable
   * @brief StateMachine に必要な情報を格納する構造体
   */
  struct AXFStateTable {
    const AXFActor::StateTable* stateTable;  ///< 状態テーブルのポインタ
  };

  std::vector<AXFStateTable> m_stateTable;     ///< 状態遷移テーブル
};

#endif /* AXFSTATEMACHINEINTERNAL_HPP_ */
