/**
 * @file   AXHIridix.hpp
 * @brief  Header file of Iridix Class
 * @par    Programming Language
 *         C++
 */
#ifndef AXH_IRIDIX_H
#define AXH_IRIDIX_H

/**
 * @enum	AXH_IRIDIX_MODE
 * @brief	IRIDIX設定モード定義
 */
typedef enum
{
  AXH_Iridix_Mode_Init = 0,                                 ///< IRIDIX初期化モード
  AXH_Iridix_Mode_Run                                       ///< IRIDIX通常モード
} AXH_IRIDIX_MODE;

/**
 * @class AXHIridix
 * @brief Iridix Class
 * 
 */
class AXHIridix {
 private:
  /**
   * @brief strength設定値
   */
  unsigned int Strength;
  /**
   * @brief スロープ最小値
   */
  unsigned int MinSlope;
  /**
   * @brief スロープ最大値
   */
  unsigned int MaxSlope;
  /**
   * @brief Variance最大値
   */
  unsigned int Variance;
  /**
   * @n Frame Widthレジスタ
   * @n Frame Heightレジスタ
   * @n (シミュレータ用ダミー変数)
   */
  unsigned int *p_IrLen;
  /**
   * @brief Ditheringレジスタ(シミュレータ用ダミー変数)
   */
  unsigned int *p_IrDither;
  /**
   * @n Variance controlレジスタ
   * @n Slope Min Limitレジスタ
   * @n Slope Max Limitレジスタ
   * @n 各レジスタ参照先
   */
  unsigned int *p_IrSlope;
  /**
   * @brief Black Level/White Levelレジスタ参照先
   */
  unsigned int *p_IrLevel;
  /**
   * @brief iridix Control0/iridix Control1レジスタ参照先
   */
  unsigned int *p_IrControl;
  /**
   * @brief iridix Strengthレジスタ参照先
   */
  unsigned int *p_IrStrength;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrLutFiAdr;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrLutFiData;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrFwdPerceptLutEn;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrFwdPerceptLutAdr;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrFwdPerceptLutData;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrRevPerceptLutEn;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrRevPerceptLutAdr;
  /**
   * @brief ルックアップテーブル参照先
   */
  unsigned int *p_IrRevPerceptLutData;
  /**
   * @brief xxxxレジスタ参照先
   */
  unsigned int *p_IrRGGBStart;
  /**
   * @brief xxxxレジスタ参照先
   */
  unsigned int *p_IrWBGain0;
  /**
   * @brief xxxxレジスタ参照先
   */
  unsigned int *p_IrWBGain1;
  /**
   * @brief xxxxレジスタ参照先
   */
  unsigned int *p_IrIFControl;

 public:
  /**
   * @brief  オブジェクトを初期化する
   */
  AXHIridix();

  /**
   * @brief  オブジェクトを開放する
   */
  ~AXHIridix();

  /**
   * @brief strength設定値を取得する
   *
   * @return strength設定値
   */
  unsigned char getStrength();

  /**
   * @brief strengthを設定する
   *
   * @param [in] strength 強度
   */
  void setStrength(unsigned char strength);

  /**
   * @brief スロープ最小値を取得する
   *
   * @return スロープ最小値の設定値
   */
  unsigned char getMinSlope();

  /**
   * @brief スロープ最小値を設定する
   *
   * @param [in] minSolpe スロープ最小値
   */
  void setMinSlope(unsigned char minSolpe);

  /**
   * @brief スロープ最大値を取得する
   *
   * @return スロープ最大値の設定値
   */
  unsigned char getMaxSlope();

  /**
   * @brief スロープ最大値を設定する
   *
   * @param [in] maxSlope スロープ最大値
   */
  void setMaxSlope(unsigned char maxSlope);

  /**
   * @brief スロープ最大値を取得する
   *
   * @return Varianceの設定値
   */
  unsigned char getVariance();

  /**
   * @brief Varianceを設定する
   *
   * @param [in] variance スロープ最大値
   */
  void setVariance(unsigned char variance);

  /**
   * @brief Iridix関連の設定値を設定する
   *
   * @param [in] mode モード
   * @param [in] width 画像サイズ幅
   * @param [in] height 画像サイズ高さ
   */
  void Execute(AXH_IRIDIX_MODE mode, unsigned short width, unsigned short height);
};
#endif
