/**
 * @file   AXHNR.hpp
 * @brief  Header file of NR Class
 * @par    Programming Language
 *         C++
 */
#ifndef AXH_NR_H
#define AXH_NR_H

#include "AXHSpecParam.hpp"

/**
 * @class AXHNR
 * @brief NR Class
 * 
 */
class AXHNR {
 private:
  /**
   * @brief NR_EN設定値(3DNR動作モード設定)
   */
  unsigned int Mode;
  /**
   * @brief NR_MAIN_YTH設定値(3DNR：Y信号 ノイズ検知レベル設定)
   */
  unsigned char MainYTH[AXH_NR_Main_Max];
  /**
   * @brief NR_MAIN_YTH設定値(3DNR：Y信号 ノイズ削減強度設定)
   */
  unsigned char MainYIN[AXH_NR_Main_Max];
  /**
   * @brief NR_MAIN_YLV設定値(3DNR：Y信号 動き検知しきい値設定)
   */
  unsigned char MainYLV[AXH_NR_Main_Max];
  /**
   * @brief NR_MAIN_CTH設定値(3DNR：C信号 ノイズ検知レベル設定)
   */
  unsigned char MainCTH[AXH_NR_Main_Max];
  /**
   * @brief NR_MAIN_CTH設定値(3DNR：C信号 ノイズ削減強度設定)
   */
  unsigned char MainCIN[AXH_NR_Main_Max];
  /**
   * @brief NR_MAIN_CLV設定値(3DNR：C信号 動き検知しきい値設定)
   */
  unsigned char MainCLV[AXH_NR_Main_Max];
  /**
   * @brief NR_SUB_YTH設定値(2DNR：Y信号 ノイズ検知レベル設定)
   */
  unsigned char SubYTH[AXH_NR_Sub_Max];
  /**
   * @brief NR_SUB_YTH設定値(2DNR：Y信号 ノイズ削減強度設定)
   */
  unsigned char SubYIN[AXH_NR_Sub_Max];
  /**
   * @brief NR_SUB_YLV設定値(2DNR：Y信号 動き検知しきい値設定)
   */
  unsigned char SubYLV[AXH_NR_Sub_Max];
  /**
   * @brief NR_SUB_CTH設定値(2DNR：C信号 ノイズ検知レベル設定)
   */
  unsigned char SubCTH[AXH_NR_Sub_Max];
  /**
   * @brief NR_Sub_CTH設定値(2DNR：C信号 ノイズ削減強度設定)
   */
  unsigned char SubCIN[AXH_NR_Sub_Max];
  /**
   * @brief NR_SUB_CLV設定値(2DNR：C信号 動き検知しきい値設定)
   */
  unsigned char SubCLV[AXH_NR_Sub_Max];
  /**
   * @brief NR_HLENレジスタ参照先
   */
  unsigned int *p_NRHLen;
  /**
   * @brief NR_VLENレジスタ参照先
   */
  unsigned int *p_NRVLen;
  /**
   * @brief NR_BLANKレジスタ参照先
   */
  unsigned int *p_NRBlank;
  /**
   * @brief NR_ENレジスタ参照先
   */
  unsigned int *p_NREn;
  /**
   * @brief NR_MAIN_YTHxレジスタ参照先
   */
  unsigned int *p_NRMainYTH[AXH_NR_MainTH_Max];
  /**
   * @brief NR_MAIN_CTHxレジスタ参照先
   */
  unsigned int *p_NRMainCTH[AXH_NR_MainTH_Max];
  /**
   * @brief NR_MAIN_YINレジスタ参照先
   */
  unsigned int *p_NRMainYIN[AXH_NR_MainTH_Max];
  /**
   * @brief NR_MAIN_CINレジスタ参照先
   */
  unsigned int *p_NRMainCIN[AXH_NR_MainIN_Max];
  /**
   * @brief NR_SUB_YTHレジスタ参照先
   */
  unsigned int *p_NRSubYTH;
  /**
   * @brief NR_SUB_CTHレジスタ参照先
   */
  unsigned int *p_NRSubCTH;
  /**
   * @brief NR_SUB_YINレジスタ参照先
   */
  unsigned int *p_NRSubYIN;
  /**
   * @brief NR_SUB_CINレジスタ参照先
   */
  unsigned int *p_NRSubCIN;
  /**
   * @brief NR_MAIN_YLVxレジスタ参照先
   */
  unsigned int *p_NRMainYLV[AXH_NR_MainLV_Max];
  /**
   * @brief NR_MAIN_CLVxレジスタ参照先
   */
  unsigned int *p_NRMainCLV[AXH_NR_MainLV_Max];
  /**
   * @brief NR_SUB_YLVレジスタ参照先
   */
  unsigned int *p_NRSubYLV;
  /**
   * @brief NR_SUB_CLVレジスタ参照先
   */
  unsigned int *p_NRSubCLV;

 public:
  /**
   * @brief  オブジェクトを初期化する
   */
  AXHNR();

  /**
   * @brief  オブジェクトを開放する
   */
  ~AXHNR();

  /**
   * @brief 3DNR動作モード設定値を取得する
   *
   * @return 3DNR動作モード設定値
   */
  unsigned char getMode();

  /**
   * @brief YTH(Y信号のノイズ検知レベル：3DNR)を設定する
   *
   * @param [in] mode 動作モード設定値
   */
  void setMode(unsigned char mode);

  /**
   * @brief YTH(Y信号のノイズ検知レベル：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YTH設定値
   */
  unsigned char getMainYTH(unsigned int index);

  /**
   * @brief YTH(Y信号のノイズ検知レベル：3DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] yth YTH設定値
   */
  void setMainYTH(unsigned int index, unsigned char yth);

  /**
   * @brief CTH(C信号のノイズ検知レベル：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CTH設定値
   */
  unsigned char getMainCTH(unsigned int index);

  /**
   * @brief CTH(C信号のノイズ検知レベル：3DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] cth CTH設定値
   */
  void setMainCTH(unsigned int index, unsigned char cth);

  /**
   * @brief YIN(Y信号のノイズ削減強度：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YIN設定値
   */
  unsigned char getMainYIN(unsigned int index);

  /**
   * @brief YIN(Y信号のノイズ削減強度：3DNR)設定値を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] yin YIN設定値
   */
  void setMainYIN(unsigned int index, unsigned char yin);

  /**
   * @brief CIN(C信号のノイズ削減強度：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CIN設定値
   */
  unsigned char getMainCIN(unsigned int index);

  /**
   * @brief CIN(C信号のノイズ削減強度：3DNR)設定値を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] cin CIN設定値
   */
  void setMainCIN(unsigned int index, unsigned char cin);

  /**
   * @brief YTH(Y信号のノイズ検知レベル：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YTH設定値
   */
  unsigned char getSubYTH(unsigned int index);

  /**
   * @brief YTH(Y信号のノイズ検知レベル：2DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] yth YTH設定値
   */
  void setSubYTH(unsigned int index, unsigned char yth);

  /**
   * @brief CTH(C信号のノイズ検知レベル：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CTH設定値
   */
  unsigned char getSubCTH(unsigned int index);

  /**
   * @brief CTH(C信号のノイズ検知レベル：2DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] cth CTH設定値
   */
  void setSubCTH(unsigned int index, unsigned char cth);

  /**
   * @brief YIN(Y信号のノイズ削減強度：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YIN設定値
   */
  unsigned char getSubYIN(unsigned int index);

  /**
   * @brief YIN(Y信号のノイズ削減強度：2DNR)設定値を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] yin YIN設定値
   */
  void setSubYIN(unsigned int index, unsigned char yin);

  /**
   * @brief CIN(C信号のノイズ削減強度：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CIN設定値
   */
  unsigned char getSubCIN(unsigned int index);

  /**
   * @brief CIN(C信号のノイズ削減強度：2DNR)設定値を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] cin CIN設定値
   */
  void setSubCIN(unsigned int index, unsigned char cin);

  /**
   * @brief YLV(Y信号の動き検知しきい値：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YLV設定値
   */
  unsigned char getMainYLV(unsigned int index);

  /**
   * @brief YLV(Y信号の動き検知しきい値：3DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] ylv YLV設定値
   */
  void setMainYLV(unsigned int index, unsigned char ylv);

  /**
   * @brief CLV(C信号の動き検知しきい値：3DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CLV設定値
   */
  unsigned char getMainCLV(unsigned int index);

  /**
   * @brief CLV(C信号の動き検知しきい値：3DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] clv CLV設定値
   */
  void setMainCLV(unsigned int index, unsigned char clv);

  /**
   * @brief YTH(Y信号の動き検知しきい値：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return YLV設定値
   */
  unsigned char getSubYLV(unsigned int index);

  /**
   * @brief YLV(Y信号の動き検知しきい値：2DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] ylv YLV設定値
   */
  void setSubYLV(unsigned int index, unsigned char ylv);

  /**
   * @brief CLV(C信号の動き検知しきい値：2DNR)設定値を取得する
   *
   * @param [in] index インデックス番号
   * @return CTH設定値
   */
  unsigned char getSubCLV(unsigned int index);

  /**
   * @brief CLV(C信号の動き検知しきい値：2DNR)を設定する
   *
   * @param [in] index インデックス番号
   * @param [in] clv CLV設定値
   */
  void setSubCLV(unsigned int index, unsigned char clv);

  /**
   * @brief NR関連の設定値を設定する
   *
   * @param [in] width 画像サイズ幅
   * @param [in] height 画像サイズ高さ
   */
  void Execute(unsigned short width, unsigned short height);

  /**
   * @brief NRの画像サイズの設定値を設定する
   *
   * @param [in] width 画像サイズ幅
   * @param [in] height 画像サイズ高さ
   */
  void ExecuteSize(unsigned short width, unsigned short height);

  /**
   * @brief NR_ENに設定値を設定する
   */
  void ExecuteMode();

  /**
   * @brief NR_ENに指定した内容を設定する
   *
   * @param [in] mode NR_EN(レジスタ)に設定する内容
   */
  void ExecuteMode(unsigned char mode);

  /**
   * @brief NR_MAIN_Y**に設定値を設定する
   */
  void ExecuteMainY();

  /**
   * @brief NR_MAIN_C**に設定値を設定する
   */
  void ExecuteMainC();

  /**
   * @brief NR_SUB_Y**に設定値を設定する
   */
  void ExecuteSubY();

  /**
   * @brief NR_SUB_C**に設定値を設定する
   */
  void ExecuteSubC();
};
#endif
