/**
 * @file   AXHOpdConfigBase.hpp
 * @brief  Header file of base class of OPD Config class.
 * @par    Programming Language
 *         C++
 */

#ifndef A_X_H_OPD_CONFIG_BASE_H
#define A_X_H_OPD_CONFIG_BASE_H

/**
 * @class AXHOpdConfigBase
 * @brief OPD Config Base class
 * 
 */
class AXHOpdConfigBase {
 private:
  /**
   * @brief 入力画像幅設定値
   */
  unsigned int Width;
  /**
   * @brief 入力画像高さ設定値
   */
  unsigned int Height;
  /**
   * @brief OPDモード設定値
   */
  int OpdMode;
  /**
   * @brief 垂直分割画素数設定値
   */
  int VerDiv;
  /**
   * @brief 水平分割画素数設定値
   */
  int HorDiv;
  /**
   * @brief 垂直分割画素サイズ設定値
   */
  int VerWSize;
  /**
   * @brief 水平分割画素サイズ設定値
   */
  int HorWSize;
  /**
   * @brief ShadeMode値
   */
  bool m_shadeMode;
  /**
   * @brief LinearMode値退避
   */
  unsigned char m_linearMode;
  /**
   * @brief モード設定レジスタ参照先
   */
  unsigned int *p_RegAVEVER_MODE_SEL;
  /**
   * @brief 分割画素数設定レジスタ参照先
   */
  unsigned int *p_RegAVEVER_W_NUM;
  /**
   * @brief 分割画素サイズ設定レジスタ参照先
   */
  unsigned int *p_RegAVEVER_W_SIZE;
  /**
   * @brief SPUコントロールレジスタ参照先
   */
  unsigned int *p_RegSPU_C;
  /**
   * @brief センサ画像データ長設定レジスタ参照先
   */
  unsigned int *p_RegSPU_DMAW0_LEN;
  /**
   * @brief 3Aデータ長設定レジスタ参照先
   */
  unsigned int *p_RegSPU_DMAW1_LEN;
  /**
   * @brief SPU/MIXデータオフセット設定参照先
   */
  unsigned int *p_RegSPU_DMAW0_OFS;
  /**
   * @brief 3Aデータオフセット設定参照先
   */
  unsigned int *p_RegSPU_DMAW1_OFS;
  /**
   * @brief MIX画像設定参照先
   */
  unsigned int *p_RegMIX_ACT;

#ifdef PARASOFT_CPPTEST
  unsigned int CPPTEST_checkValue;
#endif // PARASOFT_CPPTEST

 public:
  /**
   * @brief オブジェクトを初期化する
   */
  AXHOpdConfigBase();

  /**
   * @brief オブジェクトを初期化する
   *
   * @param [in] opdMode OPDモード設定
   * @param [in] horDiv 水平分割画素数
   * @param [in] verDiv 垂直分割画素数
   * @param [in] horWSize 水平分割画素サイズ
   * @param [in] verWSize 垂直分割画素サイズ
   */
  AXHOpdConfigBase(int opdMode, int horDiv, int verDiv, int horWSize, int verWSize);

  /**
   * @brief オブジェクトを開放する
   */
  ~AXHOpdConfigBase();

  /**
   * @brief AXHOpdConfigBaseクラスを初期化する
   */
  void Init();

  /**
   * @brief 画像幅を設定する
   */
  void setImageWidth(unsigned int width);

  /**
   * @brief 画像高さを設定する
   */
  void setImageHeight(unsigned int height);

  /**
   * @brief 画像サイズをSPU_DMAWO_LENに設定する
   *
   * @param [in] shadeMode シェーディング補正モード状態
   */
  void setImageSize(bool shadeMode);
  /**
   * @brief ImageIF動作モードを設定する
   *
   * @param [in] linearMode ImageIF動作モード状態[1:Linear動作モード/1以外:左記以外のモード]
   */
  void setLinearMode(unsigned char linearMode);

  /**
   * @brief 画像サイズをSPU_DMA0_LENに設定する
   *
   * @param [in] width 画像幅
   * @param [in] height 画像高さ
   * @param [in] shadeMode WDRモード状態
   */
  void setImageSize(unsigned int width, unsigned int height, bool shadeMode);

  /**
   * @brief OPDモードを設定する
   *
   * @param [in] opdMode OPDモード
   */
  void setOpdMode(int opdMode);

  /**
   * @brief OPDモード設定値を取得する
   *
   * @return OPDモード設定値
   */
  int getOpdMode();

  /**
   * @brief 垂直分割画素数を設定する
   *
   * @param [in] verDiv 垂直分割画素数
   */
  void setVerDiv(int verDiv);

  /**
   * @brief 垂直分割画素数設定値を取得する
   *
   * @return 垂直分割画素数設定値
   */
  int getVerDiv();

  /**
   * @brief 水平分割画素数を設定する
   *
   * @param [in] horDiv 水平分割画素数
   */
  void setHorDiv(int horDiv);

  /**
   * @brief 水平分割画素数設定値を取得する
   *
   * @return 水平分割画素数設定値
   */
  int getHorDiv();

  /**
   * @brief 垂直分割画素サイズを設定する
   *
   * @param [in] verWSize 垂直分割画素サイズ
   */
  void setVerWSize(int verWSize);

  /**
   * @brief 垂直分割画素サイズ設定値を取得する
   *
   * @return 垂直分割画素サイズ設定値
   */
  int getVerWSize();

  /**
   * @brief 水平分割画素サイズを設定する
   *
   * @param [in] horWSize 水平分割画素サイズ
   */
  void setHorWSize(int horWSize);

  /**
   * @brief 水平分割画素サイズ設定値を取得する
   *
   * @return 水平分割画素サイズ画素サイズ
   */
  int getHorWSize();

  /**
   * @brief レジスタに対してOPD設定を行う
   *
   * @param [in] cmd コマンド
   */
  void setOpdConfig(int cmd);

  /**
   * @brief レジスタに対してOPD設定を行う
   *
   * @param [in] cmd コマンド
   * @param [in] wdrMode    WDRモード状態
   */
  void setOpdConfig(int cmd, int wdrMode);

  /**
   * @brief 更新済み画像データアドレスを取得
   *
   * @return 更新済み画像データアドレス
   */
  unsigned char* getImageAddr();

};
#endif
