/**
 * @file   AXHPixcessorMemConfig.hpp
 * @brief  Header file of PixcessorMemoryConfig class.
 * @par    Programming Language
 *         C++
 */
#ifndef A_X_H_PIXCESSOR_MEM_CONFIG_H_
#define A_X_H_PIXCESSOR_MEM_CONFIG_H_

/**
 * @class  AXHPixcessorMemConfig
 * @brief  AXHPixcessorMemConfig Class
 */
class AXHPixcessorMemConfig {
public:
	/**
	 * @brief  オブジェクトを初期化する
	 */
	AXHPixcessorMemConfig();
	/**
	 * @brief  オブジェクトを開放する
	 */
	virtual ~AXHPixcessorMemConfig();

private:
	/**
	 * @brief  初期化処理
	 */
	void          init();

public:
	/**
	 * @brief  ImageIF動作モードを設定する
	 *
	 * @param  [in]  imageIFOpeMode            1:Linear動作(60fps)のモード\n
	 * @n                                      1以外:Linear動作以外(30fps)のモード
	 */
	void          setLinearMode(unsigned char imageIFOpeMode);

	/**
	 * @brief  Pixcessorメモリバンクの切り替えを行う
	 *
	 * @return 切り替え後のバンク番号\n
	 * @n      -1:バンク切り替えなし\n
	 * @n      -1以外:切り替え後バンク番号
	 */
	int           switch_pixcessor_mem_bank();

	/**
	 * @brief  第1フレーム復帰を通知する
	 *
	 * @param  [in]  set                       設定状態\n
	 * @n      trueで第1フレームへの復帰設定を行います。\n
	 * @n      次回switch_pixcessor_mem_bank実行時に、g_picNumが\n
	 * @n      INIT_PIC_NUM_FROM_ADJUSTで指定された値に初期化\n
	 * @n      されます。
	 */
	void          setBankCounterAtFirstFrame(bool set);

	/**
	 * @brief  linear時(60fps)、Pixcessorメモリバンクの切り替えを行う
	 *
	 * @return 切り替え後のバンク番号
	 */
	int           select_mem_bank_linear(unsigned char picNum);
	/**
	 * @brief  通常時(30fps)、Pixcessorメモリバンクの切り替えを行う
	 *
	 * @return 切り替え後のバンク番号\n
	 * @n      -1:バンク切り替えなし\n
	 * @n      -1以外:切り替え後バンク番号
	 */
	int           select_mem_bank_normal(unsigned char picNum);

private:
	// レジスタアドレス定義
	/**
	 * @brief  RPU 出力フォーマット設定
	 */
	unsigned int* p_SPP_SHT1;
	/**
	 * @brief  NR DMA Read Address
	 */
	unsigned int* p_NR_DMAR_OFS;
	/**
	 * @brief  RPU DMA Read0 Address
	 */
	unsigned int* p_RPU_DMAR0_OFS;
	/**
	 * @brief  RPU DMA Write1 Address
	 */
	unsigned int* p_RPU_DMAW1_OFS;
	/**
	 * @brief  IVA(ch5)用の画像ID
	 */
	unsigned int* p_REG_ISP_SCL_IMG_ID;
	/**
	 * @brief  スケーラー DMAスタートアドレス0
	 */
	unsigned int* p_REG_ISP_SCL_DMA0;
	/**
	 * @brief  スケーラー DMAスタートアドレス1
	 */
	unsigned int* p_REG_ISP_SCL_DMA1;
	/**
	 * @brief  スケーラー DMAスタートアドレス2
	 */
	unsigned int* p_REG_ISP_SCL_DMA2;
	/**
	 * @brief  スケーラー DMAスタートアドレス3
	 */
	unsigned int* p_REG_ISP_SCL_DMA3;
	/**
	 * @brief  スケーラー DMAスタートアドレス4
	 */
	unsigned int* p_REG_ISP_SCL_DMA4;
	/**
	 * @brief  バンク切り替え実行マスク
	 */
	const unsigned int EXEC_BANK_SWITCH_MASK;
	/**
	 * @brief  バンク番号１切り替えマスク
	 */
	const unsigned int SECOND_BANK_SWITCHING_MASK;
	/**
	 * @brief  画像ID用カウンタ
	 */
	unsigned char picID;
	/**
	 * @brief	Linearモード\n
	 * @n		1:Linear動作(60fps)のモード\n
	 * @n		1以外:Linear動作以外(30fps)のモード
	 */
	unsigned int  m_linearMode;
	/**
	 * @brief  第1フレーム復帰フラグ\n
	 * @n      false:復帰しない\n
	 * @n      true:復帰する
	 */
	bool m_setFirstFrame;
};

#endif /* A_X_H_PIXCESSOR_MEM_CONFIG_H_ */
