/*
 * @file   AXHWhiteBalanceGain.cpp
 * @brief  Source file of White Balance Gain Class.
 * @par    Programming Language
 *         C++
 */
#include <string>
#include <vector>
#include <list>
#include <iostream>
#include <assert.h>

#include "AXHWhiteBalanceGain.hpp"
#include "AXHCommonParam.hpp"
#include "AXHRegAddr.hpp"

extern WbgRegCtrl wbg_reg_ctrl;

WbgRegCtrl::WbgRegCtrl() {
	this->rChgFlg = false;
	this->bChgFlg = false;
	
	// 今回１レジスタにRBが存在
	this->p_rbGain = (unsigned int *)AXH_REG_WB0;	// ISP reg_0x10
	
	// r,bGain:12bit(rGain:ISP reg_0x10[11,0] / bGain:ISP reg_0x10[27,16])
	this->rGain = (*p_rbGain) & 0x00000FFF;
	this->bGain = ((*p_rbGain) >> 16) & 0x00000FFF;
}

WbgRegCtrl::~WbgRegCtrl() {
}

unsigned int WbgRegCtrl::getRegRGain() {
	return this->rGain;
}

void WbgRegCtrl::setRegRGain() {
	if (this->rChgFlg) {
		// 今回１レジスタにRBが存在(ISP reg_0x10[11,0])
		*this->p_rbGain &= 0xFFFFF000;					// Clear old value
		*this->p_rbGain |= (this->rGain & 0x00000FFF);	// Set new value
		this->rChgFlg = false;
	}
}

void WbgRegCtrl::setTmpRGain(unsigned int rgain) {
	if (rgain >= 0x400) {
		rgain = 0x3FF;
	}
	// 今回の環境は以前の環境に対し4倍する必要有
	this->rGain = rgain << 2;		//rgain;
	
	this->rChgFlg = true;	// Rゲイン変更有
}

unsigned int WbgRegCtrl::getRegBGain() {
	return bGain;
}

void WbgRegCtrl::setRegBGain() {
	if (this->bChgFlg) {
		// 今回１レジスタにRBが存在(ISP reg_0x10[27,16])
		*p_rbGain &= 0xF000FFFF;							// Clear old value
		*p_rbGain |= ((this->bGain & 0x00000FFF) << 16);	// Set new value
		this->bChgFlg = false;
	}
}

void WbgRegCtrl::setTmpBGain(unsigned int bgain) {
	if (bgain >= 0x400) {
		bgain = 0x3FF;
	}
	// 今回の環境は以前の環境に対し4倍する必要有
	this->bGain = bgain << 2;		//bgain;
	
	this->bChgFlg = true;	// Bゲイン変更有
}


AXHWhiteBalanceGain::AXHWhiteBalanceGain() {
}

AXHWhiteBalanceGain::~AXHWhiteBalanceGain() {
}

unsigned int AXHWhiteBalanceGain::getRGain() {
	return wbg_reg_ctrl.getRegRGain();
}

void AXHWhiteBalanceGain::setRGain(unsigned int rgain) {
	
	wbg_reg_ctrl.setTmpRGain(rgain);
}

unsigned int AXHWhiteBalanceGain::getBGain() {
	return wbg_reg_ctrl.getRegBGain();
}

void AXHWhiteBalanceGain::setBGain(unsigned int bgain) {
	wbg_reg_ctrl.setTmpBGain(bgain);
}
