/**
 * @file   AXMIspAdjustState.hpp
 * @brief  AXMIspAdjustクラス StateMachineオブジェクトヘッダ
 *
 *         Across ISP
 * @par    Programming Language
 *         C++
 */

/**
 * @enum  ispAdjustModelState
 * @brief StateMachine用状態定義
 */
enum ispAdjustModelState {
  STATE_WAITING = 0,                    //!<  待機中
  STATE_RUNNING,                        //!<  動作中

  STATE_MAX
};

/**
 * @enum  ispAdjustModelEvent
 * @brief StateMachine用イベント定義
 */
enum ispAdjustModelEvent {
  EVENT_PROC_GAMMA = AXFEVENT_MODEL(0), //!<  ガンマ補正設定開始
  EVENT_PROC_APERTURE,                  //!<  アパーチャ補正設定開始
  EVENT_PROC_IRIDIX_START,              //!<  IRIDIX設定開始
  EVENT_PROC_CHROMA,                    //!<  クロマサプレス設定開始
  EVENT_PROC_LENS_SHADING_START,        //!<  レンズシェーディング補正設定開始
  EVENT_PROC_NR_START,                  //!<  NR設定開始
  EVENT_PROC_SCALER_SETTINGS,           //!<  スケーラ設定開始
  EVENT_PROC_DEFECT_START,              //!<  画素欠陥補正設定開始
  EVENT_PROC_RGB_MATRIX_START,          //!<  RGBマトリックス設定開始

  EVENT_SPEC_ADDR,                      //!<  設定値保存先アドレス通知

  EVENT_CHECK_SETTINGS,
  ///!  設定値更新確認通知
};

// StateMachine用状態テーブル宣言
static const AXMIspAdjust::StateTable state_waiting[];      //!<  待機中 状態テーブル定義
static const AXMIspAdjust::StateTable state_running[];      //!<  動作中 状態テーブル定義

// StateMachine用メンバ関数宣言
/**
 * @brief 設定値保存先アドレス通知を受信し、設定値参照先にそのアドレスを設定する
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus funcSpecAddr(const void *pParam, int size);
/**
 * @brief ガンマ補正設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0000(const void *pParam, int size);
/**
 * @brief アパーチャ補正設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0100(const void *pParam, int size);
/**
 * @brief アパーチャ補正設定(輝度変調設定:LUT設定)実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0101(const void *pParam, int size);
/**
 * @brief IRIDIX設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0200(const void *pParam, int size);
/**
 * @brief クロマサプレス設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0300(const void *pParam, int size);
/**
 * @brief レンズシェーディング補正設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0400(const void *pParam, int size);
/**
 * @brief NR設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0500(const void *pParam, int size);
/**
 * @brief スケーラ機能設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0600(const void *pParam, int size);
/**
 * @brief 画素欠陥補正設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0700(const void *pParam, int size);
/**
 * @brief RGBマトリックス設定実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix0800(const void *pParam, int size);
/**
 * @brief 設定値更新確認実行
 *
 * @param [in] pParam 受信パラメータ
 * @param [in] size 受信パラメータサイズ
 * @return     実行完了状態(正常終了)
 */
ax::actorFuncStatus	matrix1000(const void *pParam, int size);
/**
 * @brief  トータルゲイン設定処理
 *
 * @param  [in]  totalGain                     トータルゲイン
 */
void					apertureTotalGainSet(BYTE totalGain);

/**
 * @brief  アパーチャ補正(正側・負側リミット設定)
 *
 * @param  [in]  limitPositive                 正側リミット設定
 * @param  [in]  limitNegative                 負側リミット設定
 */
void					apertureLimitSet(SHORT limitPositive, SHORT limitNegative);

/**
 * @brief  高域・中域ゲイン設定
 *
 * @param  [in]  midRangeRatio                 高域・中域MIX設定(高域に対する中域ゲイン比率)
 */
void					apertureOverMidRangeGainRatioSet(BYTE midRangeRatio);

/**
 * @brief  輝度変調設定:LUT設定
 *
 * @param  [in]  startSupresLuminance          暗部アパーチャ抑制開始輝度
 * @param  [in]  startStrongLuminance          高輝度側アパーチャ補正強化開始輝度
 * @param  [in]  correctSupressInc             暗部アパーチャ補正抑圧傾き
 * @param  [in]  correctStrongInc              高輝度側アパーチャ補正強化傾き
 * @param  [in]  correctStrongLimit            高輝度側アパーチャ補正強化リミット
 */
void					apertureLutTblSet(BYTE startSupresLuminance, BYTE startStrongLuminance, BYTE correctSupressInc, BYTE correctStrongInc, BYTE correctStrongLimit);

/**
 * @brief  輝度変調設定:デフォルトLUT取得
 */
void					apertureDefaultLutTblSet();

/**
 * @brief  コアリング設定
 *
 * @param  [in]  coringTh                      コアリングしきい値
 */
void					apertureCoringSet(BYTE coringTh);

/**
 * @brief  アパーチャ補正パラメータ変更チェック＆取り込み
 *
 * @retval       0                             パラメータ変更なし
 * @retval       PTN_AP_UPD_TOTALGAIN(1)       トータルゲイン設定パラメータ変更
 * @retval       PTN_AP_UPD_LIMIT(2)           リミット設定パラメータ変更
 * @retval       PTN_AP_UPD_CORING(4)          コアリング設定パラメータ変更
 * @retval       PTN_AP_UPD_MID_RATIO(8)       高域・中域ゲイン比率設定パラメータ変更
 */
USHORT					checkApertureParam();

/**
 * @brief  アパーチャ補正適応処理パラメータ変更チェック＆取り込み
 *
 * @return       パラメータ変更の有無
 */
bool					checkApertureAdapterParam();

/**
 * @brief  アパーチャ補正処理設定
 *
 * @param  [in]  updStatus                     パラメータ値変更フラグ
 */
void					setAperture(USHORT updStatus);

/**
 * @brief アパーチャ補正適応処理設定
 */
void					setApertureAdpt();

/**
 * @brief クロマサプレス色ゲイン変調設定:LUT設定
 */
void					chromaLutTblSet();

/**
 * @brief エッジクロマサプレス色ゲイン変調設定:LUT設定
 */
void                 chromaEdgeLutTblSet();

/**
 * @brief クロマサプレスパラメータ変更チェック＆取り込み
 *
 * @return	      パラメータ変更の有無
 */
bool					checkChromaParam();

/**
 * @brief エッジクロマサプレスパラメータ変更チェック＆取り込み
 *
 * @return	      パラメータ変更状態
 */
bool                 checkChromaEdgeParam();

/**
 * @brief スケーラー機能パラメータ変更チェック＆取り込み
 *
 * @retval       0                             パラメータ変更なし
 * @retval       PTN_SCL_UPD_ZOOM(1)           スケーラ機能拡大縮小設定変更
 * @retval       PTN_SCL_UPD_REVERSE(2)        スケーラ機能上下反転・ネガポジ反転設定変更
 */
USHORT					checkScalerParam();

/**
 * @brief スケーラー機能パラメータ設定
 *
 * @param	[in]  updateStatus                 パラメータ変更フラグ
 */
void					setScalerParam(USHORT updateStatus);

/**
 * @brief スケーラー機能出力イメージサイズチェック
 *
 * @param	[in]  ch                            ch番号
 * @param	[in]  in_width                      入力画像幅
 * @param	[in]  in_height                     入力画像高さ
 * @param	[in]  out_width                     出力画像幅
 * @param	[in]  out_height                    出力画像高さ
 * @return	      チェック結果
 */
bool					checkScalerOutImageSize(int ch, USHORT in_width, USHORT in_height, USHORT out_width, USHORT out_height);
/**
 * @brief IRIDIX設定値変更確認および変更処理
 *
 * @return 変更状態フラグ
 */
bool CheckIridixParam();
/**
 * @brief レンズシェーディング補正設定値変更確認および変更処理
 *
 * @return 変更状態フラグ
 */
USHORT CheckLensShadeParam();

/**
 * @brief NR設定値変更確認および変更処理
 *
 * @return 変更状態フラグ
 */
USHORT CheckNRParam();

/**
 * @brief 画素欠陥補正設定値変更確認および変更処理
 *
 * @return 変更状態フラグ
 */
USHORT CheckDefectParam();

/**
 * @brief RGBマトリックス設定値変更確認および変更処理
 *
 * @return 変更状態フラグ
 */
bool CheckRGBMatrixParam();

/**
 * @brief 設定値変更確認および変更処理
 */
void ModifyGamma();
/**
 * @brief ガンマ補正モードパラメータ変更チェック＆取り込み
 *
 * @return	      パラメータ変更の有無
 */
bool					checkGammaMode();

/**
 * @brief ガンマ補正パラメータ変更チェック＆取り込み
 *
 * @retval       0                             パラメータ変更なし
 * @retval       PTN_GM_UPD_CURVE(1)           ガンマ補正カーブ選択設定変更
 * @retval       PTN_GM_UPD_KNEE(2)            Knee設定変更
 */
USHORT					checkGammaParam();

/**
 * @brief ガンマ補正ユーザー設定パラメータ変更チェック＆取り込み
 *
 * @return	      パラメータ変更の有無
 */
bool					checkGammaUserSetParam();

int		state; 					//<! 次期ステータス格納用(setNextStateの引数)
