/**
 * @file			AXMIspGammaAlgorithm.hpp
 * @brief			AXMIspGammaAlgorithmクラスヘッダ
 *
 * @par			言語
 * 					C++
 */
#ifndef AXMISPGAMMAALGORITHM_HPP_
#define AXMISPGAMMAALGORITHM_HPP_
#include "AXMIspAdjustType.hpp"

/**
 * ガンマ値逆数計算精度
 */
const int				RECIPROCAL_PRECISION	= 1000;
/**
 * ガンマ値逆数丸め精度
 */
const int				RECIPROCAL_CUT_VALUE	= 100;
/**
 * Knee値精度
 */
const int 				KNEE_PRECISION = 1000;
/**
 * 入力画像値精度
 */
const int				X_POINT_PRECISION = 100;


/**
 * @class     AXMIspGammaAlgorithm
 * @brief     ガンマ補正アルゴリズムクラス
 *
 * @n         ガンマ補正計算時に計算アルゴリズムを提供する
 */
class AXMIspGammaAlgorithm {
public:
	/**
	 * @brief  オブジェクトを初期化する
	 *
	 * @n
	 */
	AXMIspGammaAlgorithm();
	/**
	 * @brief  オブジェクトを開放する
	 *
	 * @n
	 */
	virtual ~AXMIspGammaAlgorithm();
public:
	/**
	 * @brief	デフォルトのガンマカーブ生成
	 *
	 * @param	[in]  curvePointCnt            ガンマ補正カーブ頂点数
	 * @param	[in]  kneeBaseX                KneeポイントX(x100値)
	 * @param	[in]  kneeBaseY                KneeポイントY(x100値)
	 * @param	[in]  pixelMax                 ピクセル値最大
	 * @param	[in]  gammaValue               ガンマ値
	 * @param	[out] vertex                   ガンマ補正デフォルトカーブ頂点座標
	 * @return	      処理結果
	 */
	bool 				CreateDefaultGammaCurve(SHORT curvePointCnt, SHORT kneeBaseX, SHORT kneeBaseY, SHORT pixelMax, SHORT gammaValue, USHORT* vertex);
	/**
	 * @brief	ガンマカーブの座標変換
	 *
	 * @param	[in]  curvePointCnt            ガンマ補正カーブ頂点数
	 * @param	[in]  vertex                   ガンマ補正デフォルトカーブ頂点座標
	 * @param	[in]  kneeBaseX                KneeポイントX(x100値)
	 * @param	[in]  kneeBaseY                KneeポイントY(x100値)
	 * @param	[in]  kneeX                    変換KneeポイントX(x100値)
	 * @param	[in]  kneeY                    変換KneeポイントY(x100値)
	 * @param	[in]  pixelMax                 ピクセル値最大
	 * @param	[out] convVertex               デフォルトカーブの変換後頂点座標
	 */
	void				ConvertGammaCurve(SHORT curvePointCnt, const USHORT* vertex, SHORT kneeBaseX, SHORT kneeBaseY, SHORT kneeX, SHORT kneeY, SHORT pixelMax, USHORT* convVertex);
private:
	/**
	 * @brief	ガンマカーブの座標計算
	 *
	 * @param	[in]  curvePntIndex            ガンマ補正カーブ頂点数
	 * @param	[in]  baseXPoint               圧縮・伸長後ガンマ補正カーブX頂点１
	 * @param	[in]  nextXPoint               圧縮・伸長後ガンマ補正カーブX頂点２
	 * @param	[in]  baseYPoint               圧縮・伸長後ガンマ補正カーブY頂点１
	 * @param	[in]  nextYPoint               圧縮・伸長後ガンマ補正カーブY頂点２
	 * @param	[out] convVertex               座標変換後ガンマ補正カーブ頂点座標
	 * @param	[in]  convStartIndex           座標変換後ガンマ補正カーブ頂点計算開始インデックス
	 * @return	      座標変換後のガンマ補正カーブ頂点最終設定インデックス＋１(次回の頂点設定開始インデックス)
	 */
	int					CalcGammaCurvePoint(SHORT curvePntIndex, SHORT baseXPoint, SHORT nextXPoint, SHORT baseYPoint, SHORT nextYPoint, USHORT* convVertex, SHORT convStartIndex);

	/**
	 * @brief	線形補完座標計算
	 *
	 * @param	[in]  linearStartIndex         線形補完開始インデックス
	 * @param	[in]  linearEndIndex           線形補完終了インデックス
	 * @param	[in]  linearStart              ポイントY基準値
	 * @param	[in]  pixelMax                 ポイントY最大値
	 * @param	[out] convVertex               線形補完頂点座標
	 */
	void				CalcLinearInterpolation(SHORT linearStartIndex, SHORT linearEndIndex, SHORT linearStartValue, SHORT linearMaxValue, USHORT* convVertex);
};

#endif /* AXMISPGAMMAALGORITHM_HPP_ */
