/**
 * @file	AXMIspAwbAlgorithm.hpp
 * @brief	AXMIspAwbアルゴリズムクラス群ヘッダ
 *
 * @par		言語
 *			C++
 */


#define ENABLE_AWB_ADJUST_SPECTRUM

#ifndef AXMISPAWBALGORITHM_HPP_
#define AXMISPAWBALGORITHM_HPP_


#include	"AXMIspAwbSettings.hpp"

//	マトリックスゲイン設定ライブラリ
#include	"AXHMatrixGain.hpp"
//	ホワイトバランス設定ライブラリ
#include	"AXHWhiteBalanceGain.hpp"


/**
 * @class	CAL_COLOR
 * @brief	3Aデータ格納用クラス
 */
class	CAL_COLOR
{

public:
	/**
	 * @brief オブジェクトを初期化する
	 */
	CAL_COLOR();

	USHORT	Data[AWB_OPD_V_Div_Max][AWB_OPD_H_Div_Max][AWBColor_Max];

};


/**
 * @class	CAL_RG_BG
 * @brief	色データ格納用クラス
 */
class	CAL_RG_BG
{

public:
	/**
	 * @brief オブジェクトを初期化する
	 */
	CAL_RG_BG();

	/**
	 * @brief オブジェクトを初期化する
	 *
	 * @param [in]	init_Rg	R/G軸座標初期値
	 * @param [in]	init_Bg	B/G軸座標初期値
	 */
	CAL_RG_BG(SHORT init_Rg, SHORT init_Bg);

	SHORT	Rg[AWB_OPD_V_Div_Max][AWB_OPD_H_Div_Max];                       /*  R/G軸座標       */
	SHORT	Bg[AWB_OPD_V_Div_Max][AWB_OPD_H_Div_Max];                       /*  B/G軸座標       */
	UCHAR	Enabled[AWB_OPD_V_Div_Max][AWB_OPD_H_Div_Max];                  /*  有効/無効フラグ */
                                                                                /*    TRUE  : 有効  */
                                                                                /*    FALSE : 無効  */

};


/**
 * @class	INT_RG_BG
 * @brief	積算データ格納用クラス
 */
class INT_RG_BG
{

public:
	/**
	 * @brief オブジェクトを初期化する
	 */
	INT_RG_BG();

	LONG	bgIntData;					//	B/Gデータ積算値
	LONG	bgIntCnt;					//	B/Gデータカウント値
	LONG	rgIntData;					//	R/Gデータ積算値
	LONG	rgIntCnt;					//	R/Gデータカウント値

};


/**
 * @class	CenterPoint
 * @brief	重心座標格納クラス
 */
class CenterPoint
{

public:
	/**
	 * @brief オブジェクトを初期化する
	 */
	CenterPoint();

	/**
	 * @brief 重心座標を追加する(要素0が常に最新になるように格納する)
	 *
	 * @param [in]	bg	B/G軸座標
	 * @param [in]	rg	R/G軸座標
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	addData(SHORT bg, SHORT rg);

	ColPoint	Data[AWB_StableTerm_Max];

};


/**
 * @class	CAL_DATA
 * @brief	演算データ格納用クラス
 */
class	CAL_DATA
{

public:
  /**
   * @brief オブジェクトを初期化する
   */
  CAL_DATA();

  /**
   * @brief マトリックスデータを設定する
   *
   * @param [in]	index	象限番号
   * @param [in]	setMatrix	マトリックスデータ
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER            setMatrix(UCHAR index, ColorMatrix setMatrix);

  CAL_RG_BG         ColorData;                              //  色データ計算結果
  UCHAR             LightSourceArea[AWB_OPD_V_Div_Max][AWB_OPD_H_Div_Max];
                                                            //  光源枠判定結果
  INT_RG_BG         IntData[AWB_OPDArea_Idx_Max][AWB_WeightLightSourceMax];
                                                            //  重心計算用積算データ
  USHORT            BrightData;                             //  明るさデータ
  CenterPoint       CenterPnt;                              //  重心座標
  AWBLightSource    LightSource;                            //  光源判定結果
  AWBOPDTrustJudge  OPDTrust;                               //  OPD信頼性判定結果
  UCHAR             StableJudge;                            //  安定判定
  ColorMatrix       Matrix[AWB_ColorMatrixMax];             //  マトリックス係数(最終目標値)
  ColorMatrix       MatrixPrev[AWB_ColorMatrixMax];         //  マトリックス係数(最終目標値(前回値))
  ColorMatrix       MatrixTarget[AWB_ColorMatrixMax];       //  マトリックス係数(現在目標値)
  SHORT             MatrixDiff_CC22[AWB_ColorMatrixMax];    //  マトリックス係数(変化量)
  SHORT             MatrixDiff_CC23[AWB_ColorMatrixMax];    //  マトリックス係数(変化量)
  SHORT             MatrixDiff_CC33[AWB_ColorMatrixMax];    //  マトリックス係数(変化量)
  SHORT             MatrixDiff_CC32[AWB_ColorMatrixMax];    //  マトリックス係数(変化量)
  WBGain            WB;                                     //  ホワイトバランス(最終目標値)
  WBGain            WBPrev;                                 //  ホワイトバランス(最終目標値(前回値))
  WBGain            WBTarget;                               //  ホワイトバランス(現在目標値)
  SHORT             WBBdiff;                                //  収束制御用変化量(Bゲイン)
  SHORT             WBRdiff;                                //  収束制御用変化量(Rゲイン)
  SHORT             WBcnt;                                  //  収束制御用カウンタ
  ColPoint          Tgt;                                    //  目標値
  ColPoint          TgtPrev;                                //  前回目標値
  ColPoint          TgtTmp;                                 //  仮目標値

};


/**
 * @class JudgeAreaFrame
 * @brief エリア判定枠座標格納クラス
 */
class JudgeAreaFrame
{

public:
  /**
   * @brief オブジェクトを初期化する
   */
  JudgeAreaFrame();

  SHORT  x_min;                         //  水平方向最小座標
  SHORT  x_max;                         //  水平方向最大座標
  SHORT  y_min;                         //  垂直方向最小座標
  SHORT  y_max;                         //  垂直方向最大座標

};


/**
 * @class  AXMIspAwbAlgorithmColorData
 * @brief  色データ計算処理クラス
 *
 * @n      色データ計算処理を提供する
 */
class	AXMIspAwbAlgorithmColorData
{

public:
	/**
	 * @brief 色データ計算処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値
	 * @param [in]	p_img			画像データ先頭アドレス
	 * @param [out]	ImageData		画像データ(R, Gr, Gb, B)
	 * @param [out]	ColorData		画像データ(B/G, R/G)
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	Exec(AXMIspAwbSettings& AWBSettings, UCHAR *p_img, CAL_COLOR& ImageData, CAL_RG_BG& ColorData);

private:
	//	色データパターン取得

	/**
	 * @brief 色データをパターン(R/Gr, Gb/B)で取得する
	 *
	 * @param [in]	p_img	色データ取得先の先頭アドレス
	 * @param [in]	v		垂直座標
	 * @param [in]	h		水平座標
	 * @param [in]	col		先頭の色番号
	 * @param [out]	ImageData	画像データ(R, Gr, Gb, B)
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	getColorData(UCHAR *p_img, INT v, INT h, INT col, CAL_COLOR& ImageData);

};


/**
 * @class  AXMIspAwbAlgorithmOPDMaskedColorData
 * @brief  OPDマスク処理クラス
 *
 * @n      OPDマスク処理を提供する
 */
class	AXMIspAwbAlgorithmOPDMaskedColorData
{

public:
	/**
	 * @brief OPDマスク処理を提供する
	 *
	 * @param [in]	AWBSettings	AWB設定値
	 * @param [in,out]	ColorData	画像データ(B/G, R/G)
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	Exec(AXMIspAwbSettings& AWBSettings, CAL_RG_BG& ColorData);

};


/**
 * @class	AXMIspAwbAlgorithmJudgedLightSource
 * @brief	判定用光源格納用クラス
 */
class	AXMIspAwbAlgorithmJudgedLightSource
{

public:
  /**
   * @brief オブジェクトを初期化する
   */
  AXMIspAwbAlgorithmJudgedLightSource();

  /**
   * @brief オブジェクトを初期化する
   *
   * @param [in]	AWBSettings	AWB設定値
   */
  AXMIspAwbAlgorithmJudgedLightSource(AXMIspAwbSettings& AWBSettings);

  /**
   * @brief 判定用の光源データを設定する
   *
   * @param [in] AWBSettings AWB設定値
   */
  void resetJudgedLightSourceData(AXMIspAwbSettings& AWBSettings);

  /**
   * @brief モード別に必要な光源データを設定する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in]	index	光源index番号
   * @param [in]	num	光源番号
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER	setLightSourceData(AXMIspAwbSettings& AWBSettings, AWBLightSourceIdx index, AWBLightSource num);

  ATWMode         Mode;                                     //	 ATWモード
  INT             Num;                                      //  光源枠数
  AreaColorPoint  Area[AWBLightSource_Idx_Max];             //  光源枠座標
  JudgeAreaFrame  AreaFrm[AWBLightSource_Idx_Max];          //  判定対象区間
  ColPoint        Default[AWBLightSource_Idx_Max];          //  光源デフォルト座標

//private:
  /**
   * @brief 光源枠番号からデフォルト座標番号に変換する
   *
   * @param [in] LightSource 光源枠番号
   * @return  デフォルト座標番号
   */
  AWBLightDefault LightSourceToLightDefault(AWBLightSource LightSource);

};


/**
 * @brief  光源枠判定処理クラス
 *
 * @n      光源枠判定処理を提供する
 */
class	AXMIspAwbAlgorithmLightSourceJudgedColorData
{

public:
	/**
	 * @brief 光源枠判定処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in,out]	CalData	演算データ群
	 * @param [in,out]	JudgeLightSource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	Exec(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

};


/**
 * @class  AXMIspAwbAlgorithmWeightedColorData
 * @brief  重み付け処理クラス
 *
 * @n      重み付け処理を提供する
 */
class	AXMIspAwbAlgorithmWeightedColorData
{

public:
	/**
	 * @brief 一連の重み付け処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in,out]	CalData	演算データ群
	 * @param [in,out]	JudgeLightSource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	Exec(AXMIspAwbSettings& AWBSettings, CAL_DATA &CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

	/**
	 * @brief OPDエリアごとに積算処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in,out]	CalData	演算データ群
	 * @param [in]	JudgeLightsource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	ProcOPDArea(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightsource);

//private:
	/**
	 * @brief OPDエリア座標を取得する
	 *
	 * @param [in]	index	インデックス番号
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [out]	x_st	x軸(B/G)開始座標
	 * @param [out]	x_en	x軸(B/G)終了座標
	 * @param [out]	y_st	y軸(R/G)開始座標
	 * @param [out]	y_en	y軸(R/G)終了座標
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	GetOPDAreaPoint(INT index, AXMIspAwbSettings& AWBSettings, USHORT& x_st, USHORT& x_en, USHORT& y_st, USHORT& y_en);

private:
	/**
	 * @brief OPD重み付け処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in,out]	CalData	演算データ群
	 * @param [in]	JudgeLightsource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	ProcOPDWeight(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightsource);

	/**
	 * @brief OPD枠重み付け処理を実行する
	 *
	 * @param [in]	index	OPDエリア番号
	 * @param [in]	WeightMode	重み番号
	 * @param [in,out]	CalData	演算データ群
	 * @param [in]	JudgeLightSource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	OPDAreaMask(AWB_OPDArea index, AWBWeightMode WeightMode, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

	/**
	 * @brief 光源枠重み計算処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in]	BrightData	明るさデータ
	 * @param [in,out]	JudgeLightSource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	CalcLightSourceWeight(AXMIspAwbSettings& AWBSettings, USHORT BrightData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

	/**
	 * @brief 光源枠重み付け処理を実行する
	 *
	 * @param [in]	AWBSettings	AWB設定値群
	 * @param [in,out]	CalData	演算データ群
	 * @param [in]	JudgeLightSource	モード別光源データ群
	 * @retval AWB_E_OK	正常終了
	 * @retval AWB_E_ER	異常終了
	 */
	AWB_ER	ProcLightSource(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

};


/**
 * @class  AXMIspAwbAlgorithmBalanceCenterData
 * @brief  AWB重心計算処理クラス
 *
 * @n      重心計算処理を提供する
 */
class	AXMIspAwbAlgorithmBalanceCenterData
{

public:
  /**
   * @brief 重心計算処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]  JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER	Exec(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief WB調整モードの重心計算処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]  JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER	ExecAdjustMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

};


/**
 * @class  AXMIspAwbAlgorithmAWBTarget
 * @brief  AWB調整/収束制御処理クラス
 *
 * @n      調整/収束制御処理を提供する
 */
class	AXMIspAwbAlgorithmAWBTarget
{

public:
  /**
   * @brief ATWモードの調整/収束制御処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ExecATWMode(
      AXMIspAwbSettings& AWBSettings,
      CAL_DATA& CalData,
      AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource
      );
  /**
   * @brief ワンプッシュモードの収束制御処理を実行する
   *
   * @param [in] AWBSettings AWB設定値群
   * @param [in,out] CalData 演算データ群
   * @param [in] JudgeLightSource モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ExecOnePushMode(
      AXMIspAwbSettings& AWBSettings,
      CAL_DATA& CalData,
      AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource
      );
  /**
   * @brief マトリックス係数を算出する
   *
   * @param [in] index 要素番号
   * @param [in] LightSource 光源
   * @param [in] AWBSettings AWB設定値群
   * @return  マトリックス係数
   */
  ColorMatrix CalcMatrix(INT index, AWBLightSource LightSource, AXMIspAwbSettings& AWBSettings);

private:
  /**
   * @brief OPD信頼性判定処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcOPDTrust(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief 光源判定処理を実行する
   *
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   */
  void      ProcLightSource(CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整のための除外区間を決定する
   *
   * @param [in]	index	判定用光源の光源番号
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @return	除外区間
   */
  JudgeAreaFrame ProcMgGDirect_getJudgeAreaFrame(INT index, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する(ATW標準モード用)
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect_StandardMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する(シーンモード:屋内モード用)
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect_SceaneIndoorMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する(シーンモード:屋内蛍光灯モード用)
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect_SceaneIndoorCWFMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する(シーンモード:屋外モード用)
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect_SceaneOutdoorMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整処理を実行する(LEDモード用)
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @param [in]	JudgeLightSource	モード別光源データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMgGDirect_LEDMode(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData, AXMIspAwbAlgorithmJudgedLightSource& JudgeLightSource);

  /**
   * @brief Mg-G方向調整において、低色温度側の座標を算出する（低色温度側領域調整）
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in]	p1	デフォルト座標番号(開始座標)
   * @param [in]	p2	デフォルト座標番号(終了座標)
   * @param [in]	rg	調整前R/G軸座標
   * @return	調整後B/G軸座標
   */
  SHORT     CalcMgGDirectLowColTemp(AXMIspAwbSettings& AWBSettings, AWBLightDefault p1, AWBLightDefault p2, SHORT& rg);

  /**
   * @brief Mg-G方向調整において、高色温度側の座標を算出する（高色温度側領域調整）
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in]	p1	デフォルト座標番号(開始座標)
   * @param [in]	p2	デフォルト座標番号(終了座標)
   * @param [in]	bg	調整前B/G軸座標
   * @return	調整後B/G軸座標
   */
  SHORT     CalcMgGDirectHighColTemp(AXMIspAwbSettings& AWBSettings, AWBLightDefault p1, AWBLightDefault p2, SHORT& bg);

  /**
   * @brief Mg-G方向調整において、蛍光灯領域の座標を算出する（蛍光灯領域調整）
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in]	p1	デフォルト座標番号(開始座標)
   * @param [in]	p2	デフォルト座標番号(終了座標)
   * @param [in]	pnt	調整前座標
   * @return	調整後座標
   */
  ColPoint  CalcMgGDirectF(AXMIspAwbSettings& AWBSettings, AWBLightDefault p1, AWBLightDefault p2, ColPoint pnt);

  /**
   * @brief Mg-G方向調整において、ナトリウムランプ領域の座標を算出する（ナトリウム領域調整）
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in]	p1	デフォルト座標番号(開始座標)
   * @param [in]	p2	デフォルト座標番号(終了座標)
   * @param [in]	bg	調整前B/G軸座標
   * @return	調整後R/G軸座標
   */
  SHORT     CalcMgGDirectNaLamp(AXMIspAwbSettings& AWBSettings, AWBLightDefault p1, AWBLightDefault p2, SHORT bg);

  /**
   * @brief オフセット調整を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcOffset(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData);

  /**
   * @brief 安定性/再評価処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcStable(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData);

  /**
   * @brief WB/マトリックス決定処理を実行する
   *
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcMatrix(AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData);

  /**
   * @brief 収束変位量計算処理を実行する
   *
   * @param [in]  state       現在のステータス(STATE_ATWMode or STATE_OnePush)
   * @param [in]	AWBSettings	AWB設定値群
   * @param [in,out]	CalData	演算データ群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    ProcConvergence(UCHAR state, AXMIspAwbSettings& AWBSettings, CAL_DATA& CalData);

};


/**
 * @class  AXMIspAwbAlgorithmCurrentControlValue
 * @brief  ゲイン/マトリックス設定処理クラス
 *
 * @n      ゲイン/マトリックス設定処理を提供する
 */
class	AXMIspAwbAlgorithmCurrentControlValue
{

public:
  /**
   * @brief ゲイン/マトリックス設定処理を実行する
   *
   * @param [in]	CalData	演算データ群
   * @param [in]	AWBSettings	AWB設定値群
   * @retval AWB_E_OK	正常終了
   * @retval AWB_E_ER	異常終了
   */
  AWB_ER    Exec(CAL_DATA& CalData, ::AXMIspAwbSettings& AWBSettings);

private:
  AXHMatrixGain         regMatrix;
  AXHWhiteBalanceGain   regWB;
};


/**
 * @class  AXMIspAwbAlgorithmCalibrationLibraty
 * @brief アルゴリズムの演算で使用する共通関数群クラス
 *
 * @n      アルゴリズムの演算で使用する共通関数を提供する
 */
class AXMIspAwbAlgorithmCalibrationLibrary
{

public:
  /**
   * @brief 枠内判定を行う
   *
   * @param ［in］  AreaFrm 判定用枠座標
   * @param ［in］  CPnt  判定座標
   * @param ［in］  AreaPnt  光源枠座標
   * @retval Judge_True  真
   * @retval Judge_False 偽
   */
  Judge JudgeArea(JudgeAreaFrame AreaFrm, ColPoint CPnt, AreaColorPoint AreaPnt);

private:
  /**
   * @brief 枠内判定を行うために、B/G座標(CPnt.bg)から直線1上のR/G座標を算出する
   *
   * @param ［in］  CPnt  判定座標
   * @param ［in］  AreaPnt  光源枠座標
   * @return  R/G座標
   */
  LONG JudgeArea_Calculation1(ColPoint CPnt, AreaColorPoint Area);

  /**
   * @brief 枠内判定を行うために、R/G座標(CPnt.bg)から直線2上のB/G座標を算出する
   *
   * @param ［in］  CPnt  判定座標
   * @param ［in］  AreaPnt  光源枠座標
   * @return  B/G座標
   */
  LONG JudgeArea_Calculation2(ColPoint CPnt, AreaColorPoint Area);

  /**
   * @brief 枠内判定を行うために、B/G座標(CPnt.bg)から直線3上のR/G座標を算出する
   *
   * @param ［in］  CPnt  判定座標
   * @param ［in］  AreaPnt  光源枠座標
   * @return  R/G座標
   */
  LONG JudgeArea_Calculation3(ColPoint CPnt, AreaColorPoint Area);

  /**
   * @brief 枠内判定を行うために、R/G座標(CPnt.bg)から直線4上のB/G座標を算出する
   *
   * @param ［in］  CPnt  判定座標
   * @param ［in］  AreaPnt  光源枠座標
   * @return  B/G座標
   */
  LONG JudgeArea_Calculation4(ColPoint CPnt, AreaColorPoint Area);

};


/**
 * @class  AXMIspAwbAlgorithmSpectrumAdjust
 * @brief  センサー分光バラつき調整クラス
 *
 * @n      センサー分光バラつき調整を提供する
 */
class AXMIspAwbAlgorithmSpectrumAdjust
{

public:
  /**
   * @brief 取得した座標より、座標を正規化する
   *
   * @param [in] Mode 調整種別(低色温度側調整/高色温度側調整)
   * @param [in] CPnt 正規化座標
   * @param [in] AWBSettings AWB設定値群
   */
  AWB_ER Init(AWB_SpectrumAdjust Mode, ColPoint CPnt, AXMIspAwbSettings& AWBSettings);

};

#endif /* AXMISPAWBALGORITHM_HPP_ */
