/*
 * @file	AXMIspAwbSettings.cpp
 * @brief	AWB Package
 *
 * @par		言語
 *			C++
 */

#include "AXHSpecAwbParam.hpp"
#include "AXHSpecAwbStruct.hpp"

#include "AXMCommon/AXMCommonConfig.hpp"
#include "AXMCommon/AXMCommonStruct.hpp"

#include "AXMIspAwbTypedefine.hpp"
#include "../AXMCommon/AXMCommonSetParam.hpp"
#include "AXMIspAwbCommon.hpp"								//	AXMCommonSetParam.hppとセットでincludeする必要がある
#include "AXMIspAwbOPDConfig.hpp"
#include "AXMIspAwbSettings.hpp"

extern ST_AWB_ADJUST g_AwbAdjust;


//
//	画素座標格納用クラス
//		ImgPoint
//

//
//	Constructor (Destructor : 未定義)
//
ImgPoint :: ImgPoint()
{
	h = -1;								//	水平座標
	v = -1;								//	垂直座標
};

//
//	座標設定
//		hData : 水平軸座標データ
//		vData : 垂直軸座標データ
//
AWB_ER	ImgPoint :: setData(SHORT hData, SHORT vData)
{
	AWB_ER	ret = AWB_E_OK;

	h = hData;
	v = vData;

	return ret;
};


//
//	色画像座標格納用クラス
//		ColPoint
//

//
//	Constructor (Destructor : 未定義)
//
ColPoint :: ColPoint()
{
	bg = -1;							//	B/G軸座標
	rg = -1;							//	R/G軸座標
};

//
//	色座標設定
//		rgData : R/G軸座標データ
//		bgData : B/G軸座標データ
//
AWB_ER	ColPoint :: setData(SHORT bgData, SHORT rgData)
{
	AWB_ER	ret = AWB_E_OK;

	bg = bgData;						//	B/G軸座標
	rg = rgData;						//	R/G軸座標

	return ret;
};


//
//	光源枠格納用クラス
//		AreaColorPoint
//

//
//	Constructor (Destructor : 未定義)
//
AreaColorPoint :: AreaColorPoint()
{
	ID     = AWBLightSource_None;						//	光源枠番号
	Weight = 1;											//	光源枠重み
};

//
//	光源座標 設定
//
AWB_ER	AreaColorPoint :: setData(UCHAR index, SHORT bgData, SHORT rgData)
{
	INT	ret;

	if(index < AWB_LightSourcePnt)
	{
		ret = CPnt[index].setData(bgData, rgData);
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	光源枠重み付け設定格納用クラス
//		LightSourceWeight
//

//
//	Constructor (Destructor : 未定義)
//
LightSourceWeight :: LightSourceWeight()
{
	WeightL = 0;						// 低輝度時重み
	WeightH = 0;						// 高輝度時重み
	LimitL  = 0;						// 低輝度判定上限
	LimitH  = 0;						// 低輝度判定下限
};


//
//	WBバランス格納用クラス
//		WBGain
//

//
//	Constructor (Destructor : 未定義)
//
WBGain :: WBGain()
{
	RGain = AWB_WB_RGain_Init;		// Rホワイトバランス
	BGain = AWB_WB_BGain_Init;		// Bホワイトバランス
};


//
//	カラーマトリックス調整係数格納用クラス
//		AdjustMatrix
//

//
//	Constructor (Destructor : 未定義)
//
AdjustMatrix :: AdjustMatrix()
{
	AJCF11 = AWB_CbGain_Init;			// Cbクロマゲイン調整係数
	AJCF12 = 0x0000;					// Cbマトリックスゲイン調整係数
	AJCF21 = 0x0000;					// Crマトリックスゲイン調整係数
	AJCF22 = AWB_CrGain_Init;			// Crクロマゲイン調整係数
};


//
//	カラーマトリックス係数格納用クラス
//		ColorMatrix
//

//
//	Constructor (Destructor : 未定義)
//
ColorMatrix :: ColorMatrix()
{
  CC22 = AWB_CbGain_Init;               //  Cbクロマゲイン
  CC23 = 0x0000;                        //  Cbマトリックスゲイン
  CC33 = AWB_CrGain_Init;               //  Crクロマゲイン
  CC32 = 0x0000;                        //  Crマトリックスゲイン
};


//
//	[AWB-5]全引込み実行設定値格納用クラス
//		AXMIspAwbSettingsOnePush
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsOnePush :: AXMIspAwbSettingsOnePush()
{
  Start = 0;                            //  全引込み実行レジスタ
  //  使用OPDエリア(全範囲指定)
  OPDArea[AWBOnePushPnt_Start].setData(AWBOnePushPnt_Start_H, AWBOnePushPnt_Start_V);
  OPDArea[AWBOnePushPnt_End].setData(AWBOnePushPnt_End_H, AWBOnePushPnt_End_V);
  Range = AWBOnePushRange_ALL;          //  光源引込み範囲(全範囲)
  Speed = 0;                            //  引込み速度
};

//
//	実行レジスタ 設定
//
AWB_ER	AXMIspAwbSettingsOnePush :: setStart(UCHAR& data)
{
	AWB_ER	ret = AWB_E_OK;

	Start = data;
	data = 0;                             //  実行レジスタの内容をクリアする

	return ret;
};

//
//	使用OPDエリア 設定
//
AWB_ER	AXMIspAwbSettingsOnePush :: setOPDArea(AWBOnePushPnt index, ImgPoint data)
{
  AWB_ER ret = AWB_E_OK;

  if(index == AWBOnePushPnt_Start)
  {
    //  開始座標確認
    if((data.h < OPDArea[AWBOnePushPnt_End].h) && (data.v < OPDArea[AWBOnePushPnt_End].v))
    {
      //  開始座標更新
      OPDArea[index].setData(data.h, data.v);
    }
    else
    {
    	ret = AWB_E_ER;
    }
  }
  else if(index == AWBOnePushPnt_End)
  {
    //  終了座標確認
    if((OPDArea[AWBOnePushPnt_Start].h < data.h) && (OPDArea[AWBOnePushPnt_Start].v < data.v))
    {
      //  終了座標更新
      OPDArea[index].setData(data.h, data.v);
    }
    else
    {
    	ret = AWB_E_ER;
    }
  }
  else
  {
    return AWB_E_ER;
  }

  //  座標設定確認
  if((OPDArea[AWBOnePushPnt_End].h <= OPDArea[AWBOnePushPnt_Start].h) &&
     (OPDArea[AWBOnePushPnt_End].v <= OPDArea[AWBOnePushPnt_Start].v))
  {
    //  デフォルト座標に戻す
    OPDArea[AWBOnePushPnt_Start].setData(AWBOnePushPnt_Start_H, AWBOnePushPnt_Start_V);
    OPDArea[AWBOnePushPnt_End].setData(AWBOnePushPnt_End_H  , AWBOnePushPnt_End_V);
    ret = AWB_E_ER;
  }

  return ret;
}

//
//	光源引込み範囲 設定
//
AWB_ER	AXMIspAwbSettingsOnePush :: setRange(AWBOnePushRange data)
{
  AWB_ER	ret = AWB_E_OK;

  if(data < AWBOnePushRange_Max)
  {
	Range = data;
  }
  else
  {
	ret = AWB_E_ER;
  }

	return ret;
}

//
//	引込み速度 設定
//
AWB_ER	AXMIspAwbSettingsOnePush :: setSpeed(UCHAR data)
{
	AWB_ER	ret = AWB_E_OK;

	Speed = data;

	return	ret;
}


//
//	[AWB-6]AWB光源枠引込み範囲設定値格納用クラス
//		AXMIspAwbSettingsLightRange
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsLightRange :: AXMIspAwbSettingsLightRange()
{
	LTempAdjust = 0;					//	低色温度側引込み範囲
	HTempAdjust = 0;					//	高色範囲側引込み範囲
// CPnt									//	引込みポイント座標
	MgDirAdjust = 0;					//	Mg方向引込み範囲
	GDirAdjust = 0;						//	G方向引込み範囲
};

//
//	低色温度側引込み範囲 設定
//
AWB_ER	AXMIspAwbSettingsLightRange :: setLTempAdjust(USHORT data)
{
  if((AWB_LTempLightRangeMin <= data) && (data <= AWB_LTempLightRangeMax)) {
    LTempAdjust = data;
  }
  else {
    return AWB_E_ER;
  }

  return AWB_E_OK;
}

//
//	高色温度側引込み範囲 設定
//
AWB_ER	AXMIspAwbSettingsLightRange :: setHTempAdjust(USHORT data)
{
  if((AWB_HTempLightRangeMin <= data) && (data <= AWB_HTempLightRangeMax)) {
    HTempAdjust = data;
  }
  else {
    return AWB_E_ER;
  }

  return AWB_E_OK;
}

//
//	Mg方向引込み範囲 設定
//
AWB_ER	AXMIspAwbSettingsLightRange :: setMgDirAdjust(USHORT data)
{
	AWB_ER	ret = AWB_E_OK;

	MgDirAdjust = data;

	return ret;
}

//
//	G方向引込み範囲 設定
//
AWB_ER	AXMIspAwbSettingsLightRange :: setGDirAdjust(USHORT data)
{
	AWB_ER	ret = AWB_E_OK;

	GDirAdjust = data;

	return ret;
}


//
//	[AWB-7]WBオフセット調整設定値格納用クラス
//		AXMIspAwbSettingsOffset
//

//
//	オフセット座標設定
//
AWB_ER	AXMIspAwbSettingsOffset :: setOffset(AWBLightSource index, SHORT bg, SHORT rg)
{
	AWB_ER	ret;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		ret = Pnt[index].setData(bg, rg);
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-8]各光源枠カラーマトリックス調整設定値格納用クラス
//		AXMIspAwbSettingsAdjustMatrix
//

//
//	Cbクロマゲイン調整係数 設定
//
AWB_ER	AXMIspAwbSettingsAdjustMatrix :: setAJCF11(AWBLightSource index, USHORT AJCF11)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Matrix[index].AJCF11 = AJCF11;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Cbマトリックス調整係数 設定
//
AWB_ER	AXMIspAwbSettingsAdjustMatrix :: setAJCF12(AWBLightSource index, USHORT AJCF12)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Matrix[index].AJCF12 = AJCF12;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Crマトリックスゲイン調整係数 設定
//
AWB_ER	AXMIspAwbSettingsAdjustMatrix :: setAJCF21(AWBLightSource index, USHORT AJCF21)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Matrix[index].AJCF21 = AJCF21;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Crクロマゲイン調整係数
//
AWB_ER	AXMIspAwbSettingsAdjustMatrix :: setAJCF22(AWBLightSource index, USHORT AJCF22)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Matrix[index].AJCF22 = AJCF22;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-9]マニュアルモード時ゲイン指定設定値格納用クラス
//		AXMIspAwbSettingsWB
//

//
//	WB Rゲイン設定
//
AWB_ER	AXMIspAwbSettingsWB :: setRGain(USHORT	RGainData)
{
	AWB_ER	ret = AWB_E_OK;

	RGain = RGainData;

	return ret;
}

//
//	WB Bゲイン設定
//
AWB_ER	AXMIspAwbSettingsWB :: setBGain(USHORT	BGainData)
{
	AWB_ER	ret = AWB_E_OK;

	BGain = BGainData;

	return ret;
}


//
//	[AWB-10]AWB遷移特性設定
//		AXMIspAwbSettingsConvergence
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsConvergence :: AXMIspAwbSettingsConvergence()
{
	Speed = 0;							//	引込み速度
	LightTH = 0;						//	照明変化検知閾値
	StableTH = 0;						//	安定性確認閾値
	StableTerm = 0;						//	安定性確認期間
};

//
//	引込み速度 設定
//
AWB_ER	AXMIspAwbSettingsConvergence :: setSpeed(USHORT SpeedData)
{
	AWB_ER	ret = AWB_E_OK;

	Speed = SpeedData;

	return ret;
}

//
//	照明変化検知閾値 設定
//
AWB_ER	AXMIspAwbSettingsConvergence :: setLightTHData(USHORT LightTHData)
{
	AWB_ER ret = AWB_E_OK;

	LightTH = LightTHData;

	return ret;
}

//
//	安定性確認閾値 設定
//
AWB_ER	AXMIspAwbSettingsConvergence :: setStableTHData	(USHORT StableTHData)
{
	AWB_ER ret = AWB_E_OK;

	StableTH = StableTHData;

	return ret;
}

//
//	安定性確認期間 設定
//
AWB_ER	AXMIspAwbSettingsConvergence :: setStableTerm(USHORT StableTermData)
{
	AWB_ER ret = AWB_E_OK;

	if((AWB_StableTerm_Min <= StableTermData) && (StableTermData <= AWB_StableTerm_Max))
	{
	  StableTerm = StableTermData;
	}
	else
	{
	  ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-11]光源枠重み付け設定値格納用構造体
//		AXMIspAwbSettingsLightSourceWeight
//

//
//	低輝度時重み設定
//
AWB_ER	AXMIspAwbSettingsLightSourceWeight :: setWeightL(AWBLightSource index, UCHAR WeightLData)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Weight[index].WeightL = WeightLData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	高輝度時重み設定
//
AWB_ER	AXMIspAwbSettingsLightSourceWeight :: setWeightH(AWBLightSource index, UCHAR WeightHData)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Weight[index].WeightH = WeightHData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	低輝度判定上限値設定
//
AWB_ER	AXMIspAwbSettingsLightSourceWeight :: setLimitL(AWBLightSource index, USHORT LimitLData)
{
	AWB_ER	ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Weight[index].LimitL = LimitLData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	高輝度判定下限設定
//
AWB_ER	AXMIspAwbSettingsLightSourceWeight :: setLimitH(AWBLightSource index, USHORT LimitHData)
{
	AWB_ER ret = AWB_E_OK;

	if((index != AWBLightSource_None) && (index < AWBLightSource_Max))
	{
		Weight[index].LimitH = LimitHData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-13]OPD設定値格納用クラス
//		AXMIspAwbSettingsOPDSet
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsOPDSet :: AXMIspAwbSettingsOPDSet()
{
	HDiv   = AWB_OPD_H_Div;				//	水平分割数
	VDiv   = AWB_OPD_V_Div;				//	垂直分割数
	HLimit = 0xFFFF;					//	積算対象信号上限
	LLimit = 0x0000;					//	積算対象信号下限
};

//
//	水平分割数 設定
//
AWB_ER	AXMIspAwbSettingsOPDSet :: setHDiv(USHORT HDivData)
{
	AWB_ER	ret = AWB_E_OK;

	HDiv = HDivData;

	return ret;
}

AWB_ER	AXMIspAwbSettingsOPDSet :: setVDiv(USHORT VDivData)
{
	AWB_ER	ret = AWB_E_OK;

	VDiv = VDivData;

	return ret;
}

//
//	積算対象信号上限 設定
//
AWB_ER	AXMIspAwbSettingsOPDSet :: setHLimit(USHORT HLimitData)
{
	AWB_ER	ret = AWB_E_OK;

	HLimit = HLimitData;

	return ret;
}

//
//	積算対象信号下限 設定
//
AWB_ER	AXMIspAwbSettingsOPDSet :: setLLimit(USHORT LLimitData)
{
	AWB_ER	ret = AWB_E_OK;

	LLimit = LLimitData;

	return ret;
}


//
//	[AWB-14]OPD重み付け設定格納用クラス
//		AXMIspSettingsOPDWeight
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsOPDWeight :: AXMIspAwbSettingsOPDWeight()
{
  V1 = AWB_OPDWeightV1_Init;            //  エリア座標v1
  H2 = AWB_OPDWeightH2_Init;            //  エリア座標h2
  H3 = AWB_OPDWeightH3_Init;            //  エリア座標h3
  V4 = AWB_OPDWeightV4_Init;            //  エリア座標v4
  for(INT i = 0; i < AWB_OPDArea_Idx_Max; i++)
  {
    WeightMode[i] = AWBWeightMode_1;
  }
};

//
//	エリア座標V1 設定
//
AWB_ER	AXMIspAwbSettingsOPDWeight :: setV1(USHORT V1Data)
{
	AWB_ER	ret = AWB_E_OK;

	V1 = V1Data;

	return ret;
}

//
//	エリア座標H2 設定
//
AWB_ER	AXMIspAwbSettingsOPDWeight :: setH2(USHORT H2Data)
{
	AWB_ER	ret = AWB_E_OK;

	H2 = H2Data;

	return ret;
}

//
//	エリア座標H3 設定
//
AWB_ER	AXMIspAwbSettingsOPDWeight :: setH3(USHORT H3Data)
{
	AWB_ER	ret = AWB_E_OK;

	H3 = H3Data;

	return ret;
}

//
//	エリア座標V4 設定
//
AWB_ER	AXMIspAwbSettingsOPDWeight :: setV4(USHORT V4Data)
{
	AWB_ER	ret = AWB_E_OK;

	V4 = V4Data;

	return ret;
}

//
//	重み設定
//
AWB_ER	AXMIspAwbSettingsOPDWeight :: setWeightMode(AWB_OPDArea index, AWBWeightMode WeightModeData)
{
	AWB_ER	ret = AWB_E_OK;

	if((index < AWB_OPDArea_Idx_Max) && (WeightModeData < AWBWeightMode_Max))
	{
		WeightMode[index] = WeightModeData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-15]OPD枠AWBマスク領域設定値格納用クラス
//		AXMIspAwbSettingsOPDMask
//

//
//	マスク座標設定
//		index : 座標インデックス番号
//		h     : 水平座標
//		v     : 垂直座標
//
AWB_ER	AXMIspAwbSettingsOPDMask :: setData(UCHAR index, SHORT h, SHORT v)
{
	AWB_ER	ret;

	if(index < AWB_OPDMaskSetMax)
	{
		ret = Pnt[index].setData(h, v);
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}


//
//	[AWB-16]センサー分光バラツキ調整設定値格納用クラス
//		AXMIspAwbSettingsSpectrumAdjust
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsSpectrumAdjust :: AXMIspAwbSettingsSpectrumAdjust()
{
	LColorTempStart = 0;				//	低色温度側調整起動レジスタクリア
	HColorTempStart = 0;				//	高色温度側調整起動レジスタクリア
}

AWB_ER	AXMIspAwbSettingsSpectrumAdjust :: setLowTempPoint(SHORT bg, SHORT rg)
{
	AWB_ER	ret = AWB_E_OK;

	LowTempPnt.bg = bg;
	LowTempPnt.rg = rg;

	return ret;
}

//
//	高温側調整ポイント 設定
//
AWB_ER	AXMIspAwbSettingsSpectrumAdjust :: setHighTempPoint(SHORT bg, SHORT rg)
{
	AWB_ER	ret = AWB_E_OK;

	HighTempPnt.bg = bg;
	HighTempPnt.rg = rg;

	return ret;
}


//
//	[AWB-17]光源枠詳細調整設定値格納用クラス
//		AXMIspAwbSettingsLightSource
//

//
//	光源枠設定
//
AWB_ER	AXMIspAwbSettingsLightSource :: setData(AWBLightSource index1, UCHAR index2, SHORT bg, SHORT rg)
{
  AWB_ER  ret;

  if((index1 != AWBLightSource_None) && (index1 < AWBLightSource_Rolling_Max)) {
    ret = Area[index1].setData(index2, bg, rg);
    AreaOriginal[index1].setData(index2, bg, rg);
  }
  else {
    ret = AWB_E_ER;
  }

  return ret;
}


//
//	[AWB-18]光源枠デフォルト位置設定値格納用クラス
//		AXMIspAwbSettingsLightDefault
//

//
//	光源座標設定
//
AWB_ER	AXMIspAwbSettingsLightDefault :: setData(AWBLightDefault index, SHORT bg, SHORT rg)
{
	AWB_ER	ret;

	if((index != AWBLightDefault_None) && (index < AWBLightDefault_Max))
	{
		ret = CPnt[index].setData(bg, rg);
	}
	else
	{
		ret = AWB_E_ER;
	}

	return	ret;
}


//
//	[AWB-19]OPD信頼性判定レベル設定値格納用クラス
//		AXMIspAwbSettignsOPDTrust
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsOPDTrust :: AXMIspAwbSettingsOPDTrust()
{
	TH = 0;								//	信頼性閾値
	Sel = AWBLowTrust_Hold;				//	低信頼性時処理選択
	Zero = AWBLightSource_A;			//	ゼロ時処理選択
};

//
//	信頼性閾値 設定
//
AWB_ER	AXMIspAwbSettingsOPDTrust :: setTH(USHORT THData)
{
	AWB_ER	ret = AWB_E_OK;

	TH = THData;

	return ret;
}

//
//	低信頼性時の処理選択 設定
//
AWB_ER	AXMIspAwbSettingsOPDTrust :: setSel(AWBLowTrust SelData)
{
	AWB_ER	ret = AWB_E_OK;

	if(SelData < AWBLowTrust_Max)
	{
		Sel = SelData;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	OPDデータゼロ時処理選択 設定
//
AWB_ER	AXMIspAwbSettingsOPDTrust :: setZero(AWBLightSource ZeroData)
{
	AWB_ER	ret = AWB_E_OK;

	Zero = ZeroData;

	return ret;
}


//
//	[Img-5.2.2] カラーマトリックス設定値格納用クラス
//		AXMIspAwbSettingsColorMatrix
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettingsColorMatrix :: AXMIspAwbSettingsColorMatrix()
{
	Mode = 0;							//	独立モード
	CC11 = 0x0100;						//	Yゲイン
	COF1 = 0;							//	Yオフセット
};

//
//	独立モードON/OFF 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setMode(UCHAR ModeData)
{
	AWB_ER	ret = AWB_E_OK;

	Mode = ModeData;

	return ret;
}

//
//	Cbクロマゲイン 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCC22(UCHAR index, USHORT CC22Data)
{
	AWB_ER	ret = AWB_E_OK;

	if(index < AWB_ColorMatrixMax)
	{
		Matrix[index].CC22 = CC22Data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Cbマトリックスゲイン 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCC23(UCHAR index, USHORT CC23Data)
{
	AWB_ER	ret = AWB_E_OK;

	if(index < AWB_ColorMatrixMax)
	{
		Matrix[index].CC23 = CC23Data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Crクロマゲイン 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCC33(UCHAR index, USHORT CC33Data)
{
	AWB_ER	ret = AWB_E_OK;

	if(index < AWB_ColorMatrixMax)
	{
		Matrix[index].CC33 = CC33Data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Crマトリックスゲイン 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCC32(UCHAR index, USHORT CC32Data)
{
	AWB_ER	ret = AWB_E_OK;

	if(index < AWB_ColorMatrixMax)
	{
		Matrix[index].CC32 = CC32Data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	Yゲイン 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCC11(USHORT CC11Data)
{
	AWB_ER	ret = AWB_E_OK;

	CC11 = CC11Data;

	return ret;
}

//
//	Yオフセット 設定
//
AWB_ER	AXMIspAwbSettingsColorMatrix :: setCOF1(USHORT COF1Data)
{
	AWB_ER	ret = AWB_E_OK;

	COF1 = COF1Data;

	return ret;
}


//
//	AWB関連設定値格納用親クラス
//		AXMIspAwbSettings
//

//
//	Constructor (Destructor : 未定義)
//
AXMIspAwbSettings :: AXMIspAwbSettings()
{
  AWB_Mode = AWBMode_ATWMode;                               //  AWBモード選択
  ATW_SceaneMode = ATWSceaneMode_IndoorMode;                //  ATWシーンモード選択
  ColorTemp = AWB_ManuModeColorTemp_Init;                   //  マニュアルモード時色温度指定
  LightSelect = AWBLightSelect_A;                           //  マニュアルモード時照明指定
  StopLowBright = AWB_StopLowBright_Init;                   //  ATW停止閾値
};

//
//	[AWB-1]AWBモード設定値変更
//
AWB_ER	AXMIspAwbSettings :: setAWBMode(AWBMode data)
{
	AWB_ER	ret = AWB_E_OK;

	if(data < AWBMode_Max)
	{
		AWB_Mode = data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	[AWB-2]ATWシーンモード設定値変更
//
AWB_ER	AXMIspAwbSettings :: setATWSceaneMode(ATWSceaneMode data)
{
	AWB_ER	ret = AWB_E_OK;

	if(data < ATWSceaneMode_Max)
	{
		ATW_SceaneMode = data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	[AWB-3]マニュアルモード時色温度指定設定値変更
//
AWB_ER	AXMIspAwbSettings :: setColorTemp(USHORT data)
{
	AWB_ER	ret = AWB_E_OK;

	ColorTemp = data;

	return ret;
}

//
//	[AWB-4]マニュアルモード時照明指定設定値変更
//
AWB_ER	AXMIspAwbSettings :: setLightSelect(AWBLightSelect data)
{
	AWB_ER	ret = AWB_E_OK;

	if(data < AWBLightSelect_Max)
	{
		LightSelect = data;
	}
	else
	{
		ret = AWB_E_ER;
	}

	return ret;
}

//
//	[AWB-12]ATW停止閾値設定値変更
//
AWB_ER	AXMIspAwbSettings :: setStopLowBright(USHORT data)
{
	AWB_ER	ret = AWB_E_OK;

	StopLowBright = data;

	return ret;
}

//
//  低色温度側引き込み範囲座標算出
//
void AXMIspAwbSettings :: setLTempAdjustPnt()
{
  //  設定温度から1800K-3200Kまでの区間の座標を算出する
  ::ColPoint LTempPnt = LightDefault.CPnt[AWBLightDefault_1800K];
  ::ColPoint HTempPnt = LightDefault.CPnt[AWBLightDefault_3200K];
  //  B/G方向の座標を算出する
  INT x, y;
  x = (INT)((HTempPnt.bg - LTempPnt.bg) * (LightRange.LTempAdjust - AWB_ColorTemp_1800K) /
                                        (AWB_ColorTemp_3200K - AWB_ColorTemp_1800K) + LTempPnt.bg);
  if((HTempPnt.bg - LTempPnt.bg) != 0) {
    y = (HTempPnt.rg - LTempPnt.rg) * (x - HTempPnt.bg) / (HTempPnt.bg - LTempPnt.bg) + HTempPnt.rg;
  }
  else {
    y = (INT)((HTempPnt.rg - LTempPnt.rg) * (LightRange.LTempAdjust - AWB_ColorTemp_1800K) /
                                        (AWB_ColorTemp_3200K - AWB_ColorTemp_1800K) + LTempPnt.rg);
  }
  LightRange.CPnt[AWB_LightRange_LowTemp].bg = (SHORT)x;
  LightRange.CPnt[AWB_LightRange_LowTemp].rg = (SHORT)y;
}

//
//  高色温度側引き込み範囲座標算出
//
void AXMIspAwbSettings :: setHTempAdjustPnt()
{
  //  設定温度から7000K-10000Kまでの区間の座標を算出する
  ::ColPoint LTempPnt = LightDefault.CPnt[AWBLightDefault_7000K];
  ::ColPoint HTempPnt = LightDefault.CPnt[AWBLightDefault_10000K];
  //  B/G方向の座標を算出する
  INT x, y;
  x = (INT)((HTempPnt.bg - LTempPnt.bg) * (LightRange.HTempAdjust - AWB_ColorTemp_7000K) /
    		                            (AWB_ColorTemp_10000K - AWB_ColorTemp_7000K) + LTempPnt.bg);
  if((HTempPnt.bg - LTempPnt.bg) != 0) {
    y = (HTempPnt.rg - LTempPnt.rg) * (x - HTempPnt.bg) / (HTempPnt.bg - LTempPnt.bg) + HTempPnt.rg;
  }
  else {
    y = (INT)((HTempPnt.rg - LTempPnt.rg) * (LightRange.HTempAdjust - AWB_ColorTemp_7000K) /
      		                            (AWB_ColorTemp_10000K - AWB_ColorTemp_7000K) + LTempPnt.rg);
  }
  LightRange.CPnt[AWB_LightRange_HighTemp].bg = (SHORT)x;
  LightRange.CPnt[AWB_LightRange_HighTemp].rg = (SHORT)y;
}

//
//  Mg-G方向引き込み簡易調整 補正座標算出処理
//
void AXMIspAwbSettings :: setMgGDirAdjust()
{
  //  Mg方向調整:基準座標に近づけるように調整する
  //  G方向調整:基準座標から遠ざけるように調整する
  //  [例外]水銀灯光源枠座標:G方向にのみ動かす
  //  基準座標(5800Kのbg座標と3200Kのrg座標)
  ::ColPoint stdPnt;
  stdPnt.rg = LightDefault.CPnt[AWBLightDefault_3200K].rg;
  stdPnt.bg = LightDefault.CPnt[AWBLightDefault_5800K].bg;
  //  基準座標が原点になるように光源枠座標を移動させてから演算する
  for(INT i = 0; i < AWBLightSource_Rolling_Max; i++) {
    if(i != AWBLightSource_Merculy) {
      //  水銀灯光源枠座標以外 演算
      //  調整座標判定
      //  光源枠座標の合計が
      //  B/G <= R/G (低色温度側)のとき、座標1/座標2をMg方向調整し、座標0/座標3をG方向調整する
      //  R/G <  B/G (高色温度側)のとき、座標0/座標1をMg方向調整し、座標2/座標3をG方向調整する
      INT diff = 0;
      for(INT j = 0; j < AWB_LightSourcePnt; j++) {
        diff += (LightSource.AreaOriginal[i].CPnt[j].rg - LightSource.AreaOriginal[i].CPnt[j].bg);
      }
      INT pnt_mg0, pnt_mg1, pnt_g0, pnt_g1;
      if(0 <= diff) {
        pnt_mg0 = AWB_LightSource_Pnt_1;
        pnt_mg1 = AWB_LightSource_Pnt_2;
        pnt_g0 = AWB_LightSource_Pnt_0;
        pnt_g1 = AWB_LightSource_Pnt_3;
      }
      else {
        pnt_mg0 = AWB_LightSource_Pnt_0;
        pnt_mg1 = AWB_LightSource_Pnt_1;
        pnt_g0 = AWB_LightSource_Pnt_2;
        pnt_g1 = AWB_LightSource_Pnt_3;
      }
      //  G方向調整
      if((LightRange.GDirAdjust != AWB_LightRange_Value) && (LightRange.GDirAdjust != 0)) {
        LightSource.Area[i].CPnt[pnt_g0]
                    = calcMgGDirAdjust(AWB_Adjust_Dir_G, stdPnt, LightSource.AreaOriginal[i].CPnt[pnt_g0]);
        LightSource.Area[i].CPnt[pnt_g1]
                    = calcMgGDirAdjust(AWB_Adjust_Dir_G, stdPnt, LightSource.AreaOriginal[i].CPnt[pnt_g1]);
      }
      else {
        LightSource.Area[i].CPnt[pnt_g0] = LightSource.AreaOriginal[i].CPnt[pnt_g0];
        LightSource.Area[i].CPnt[pnt_g1] = LightSource.AreaOriginal[i].CPnt[pnt_g1];
      }
      //  Mg方向調整
      if((LightRange.MgDirAdjust != AWB_LightRange_Value) && (LightRange.MgDirAdjust != 0)) {
        LightSource.Area[i].CPnt[pnt_mg0]
                    = calcMgGDirAdjust(AWB_Adjust_Dir_Mg, stdPnt, LightSource.AreaOriginal[i].CPnt[pnt_mg0]);
        LightSource.Area[i].CPnt[pnt_mg1]
                    = calcMgGDirAdjust(AWB_Adjust_Dir_Mg, stdPnt, LightSource.AreaOriginal[i].CPnt[pnt_mg1]);
      }
      else {
        LightSource.Area[i].CPnt[pnt_mg0] = LightSource.AreaOriginal[i].CPnt[pnt_mg0];
        LightSource.Area[i].CPnt[pnt_mg1] = LightSource.AreaOriginal[i].CPnt[pnt_mg1];
      }
    }
    else {
      //  水銀灯光源枠座標 演算(G方向調整演算)
      if((LightRange.GDirAdjust != AWB_LightRange_Value) && (LightRange.GDirAdjust != 0)) {
        for(INT j = 0; j < AWB_LightSourcePnt; j++) {
          LightSource.Area[i].CPnt[j] = calcMgGDirAdjust(AWB_Adjust_Dir_G, stdPnt, LightSource.AreaOriginal[i].CPnt[j]);
        }
      }
      else {
        for(INT j = 0; j < AWB_LightSourcePnt; j++) {
          LightSource.Area[i].CPnt[j] = LightSource.AreaOriginal[i].CPnt[j];
        }
      }
    }
    //  ARMとの共有エリアにもコピー
    for(INT j = 0; j < AWB_LightSourcePnt; j++) {
      g_AwbAdjust.LightSource[i].Area[j].Bg = LightSource.Area[i].CPnt[j].bg;
      g_AwbAdjust.LightSource[i].Area[j].Rg = LightSource.Area[i].CPnt[j].rg;
    }
  }
  //  光源デフォルト座標コピー(ARMとの共有エリアにコピー)
  for(INT i = 0; i < AWBLightDefault_Max; i++) {
    g_AwbAdjust.LightDefault[i].Bg = LightDefault.CPnt[i].bg;
    g_AwbAdjust.LightDefault[i].Rg = LightDefault.CPnt[i].rg;
  }
  //  調整ポイントコピー(ARMとの共有エリアにコピー)
  g_AwbAdjust.pntLowTemp.Bg = SpectrumAdjust.LowTempPnt.bg;
  g_AwbAdjust.pntLowTemp.Rg = SpectrumAdjust.LowTempPnt.rg;
  g_AwbAdjust.pntHighTemp.Bg = SpectrumAdjust.HighTempPnt.bg;
  g_AwbAdjust.pntHighTemp.Rg = SpectrumAdjust.HighTempPnt.rg;
}

//
//  Mg-G方向引き込み簡易調整 座標演算用関数
//
::ColPoint AXMIspAwbSettings :: calcMgGDirAdjust(AWB_Adjust_Dir dir, ::ColPoint stdPnt, ::ColPoint Pnt)
{
  ::ColPoint ret;
  INT bg, rg;
  INT Adjust;
  if(dir == AWB_Adjust_Dir_Mg) {
    Adjust = LightRange.MgDirAdjust;
  }
  else {
    Adjust = LightRange.GDirAdjust;
  }
  //  基準座標が原点になるように光源座標を移動する
  //  移動した光源座標で演算し、基準点からの座標になるように元に戻す
  bg = Pnt.bg - stdPnt.bg;
  bg = bg * Adjust / AWB_LightRange_Value + stdPnt.bg;
  rg = Pnt.rg - stdPnt.rg;
  rg = rg * Adjust / AWB_LightRange_Value + stdPnt.rg;
  ret.bg = (SHORT)bg;
  ret.rg = (SHORT)rg;
  return ret;
}
