/**
 * @file   AXCComm.hpp
 * @brief  AXCCommクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCCOMM_HPP_
#define AXCCOMM_HPP_

#include "AXCStdint.hpp"

#include <cstddef>

typedef UINT32_t AXComm_t;  ///< AXCommタイプ型
#define AXCOMM_TYPE_SHIFT   static_cast<AXComm_t>(24)
#define AXCOMM_TYPE_MASK    static_cast<AXComm_t>(0xffU << AXCOMM_TYPE_SHIFT) ///< AXCommタイプ判定用マスク
#define AXCOMM_TYPE_PROCESS static_cast<AXComm_t>(   1 << AXCOMM_TYPE_SHIFT) ///< プロセス間通信タイプ
#define AXCOMM_TYPE_CPU     static_cast<AXComm_t>(   2 << AXCOMM_TYPE_SHIFT) ///< CPU間通信タイプ
#define AXCOMM_PROCESS(x)   static_cast<AXComm_t>(AXCOMM_TYPE_PROCESS | (x)) ///< プロセス間通信タイプ定義マクロ
#define AXCOMM_CPU(x)       static_cast<AXComm_t>(AXCOMM_TYPE_CPU     | (x)) ///< CPU間通信タイプ定義マクロ

/**
 * @class AXCComm
 * @brief AXC層の Commクラス に対し、抽象クラスを提供する
 */
class AXCComm {
 public:
  /** @enum  commStatus
   *  @brief Comm実行完了状態
   */
  enum commStatus {
    COMM_SUCCESS = 0,   ///< 正常終了
    COMM_ERROR,         ///< 異常終了
    COMM_ERROR_OPEN,    ///< ファイルOpenエラー
    COMM_ERROR_NOINIT,  ///< AXComm未初期化
  };

  /**
   * @brief AXCCommコンストラクタ
   */
  AXCComm();

  /**
   * @brief AXCCommデストラクタ
   */
  virtual ~AXCComm();

  /**
   * @brief 初期化
   * @param [in] type AXCommタイプ
   * @return AXComm初期化状況
   * - COMM_SUCCESS 正常終了
   * - COMM_ERROR 異常終了
   */
  virtual commStatus init(AXComm_t type) = 0;

  /**
   * @brief 同期読み出し
   * @param [out] buf
   * @param [in] size
   * @return 読み出しデータサイズ(エラー時は 0 未満)
   */
  virtual int readSync(char *buf, int size) = 0;

  /**
   * @brief 同期書き込み
   * @param [in] buf
   * @param [in] size
   * @return 書き込みデータサイズ(エラー時は 0 未満)
   */
  virtual int writeSync(char *buf, int size) = 0;
};

#endif /* AXCCOMM_HPP_ */
