/**
 * @file   AXCConditionVariableFreeRTOS.hpp
 * @brief  Header file of AXC ConditionVariable for FreeRTOS.
 *
 * @par    Programming Language
 *         C++
 */
#ifndef AXCCONDITIONVARIABLEFREERTOS_HPP_
#define AXCCONDITIONVARIABLEFREERTOS_HPP_

#include "AXCConditionVariable.hpp"
#include "FreeRTOS.h"
#include "event_groups.h"

/**
 * @class AXCConditionVariableFreeRTOS
 * @brief AXFConditionVariable に対し、条件変数を使用して、スレッドの停止と再開を行う機能を提供する
 *
 * @n     [注意]AXM層は本クラスは直接使用せず、AXFConditionVariable を使用し、OS依存しないようにする。
 * @n     条件変数を使用して、スレッド実行の停止と再開を行う機能を提供する。
 * @n     使用方法:
 * @n     1. ConditionVariableインスタンスを生成する。
 * @n       int stat;
 * @n       AXCConditionVariableFreeRTOS* cond = new AXCConditionVariableFreeRTOS();
 * @n     2. ConditionVariableインスタンスの初期化を行う。
 * @n        本メソッドは、複数回呼び出してはならない。
 * @n        初期化に失敗した場合、呼び出し元の異常処理を行う。
 * @n       stat = cond.init();
 * @n       if (AXCConditionVariableFreeRTOS::COND_SUCCESS != stat) {
 * @n         ...
 * @n     3. 該当するスレッドの実行を停止し、プロセッサを手放す。
 * @n        waitに失敗した場合、呼び出し元の異常処理を行う。
 * @n       stat = cond.wait();
 * @n       if (AXCConditionVariableFreeRTOS::COND_SUCCESS != stat) {
 * @n         ...
 * @n     4. 該当するスレッドの実行を再開する。
 * @n        signalに失敗した場合、呼び出し元の異常処理を行う。
 * @n       stat = cond.signal();
 * @n       if (AXCConditionVariableFreeRTOS::COND_SUCCESS != stat) {
 * @n         ...
 * @n     5. ConditionVariableインスタンスを破棄する。
 * @n       delelte cond;
 */

/**
 * @class AXCConditionVariableFreeRTOS
 * @brief Class of AXCConditionVariableFreeRTOS
 */
class AXCConditionVariableFreeRTOS : public AXCConditionVariable {
  bool m_isInited;            ///< 初期化フラグ

 public:
  /**
   * @brief AXCConditionVariableFreeRTOSコンストラクタ
   */
  AXCConditionVariableFreeRTOS();

  /**
   * @brief AXCConditionVariableFreeRTOSデストラクタ
   */
  virtual ~AXCConditionVariableFreeRTOS();

  /**
   * @brief 条件変数の初期化
   * @n
   * @return 条件変数初期化状況
   * - COND_SUCCESS 正常終了
   * - COND_ERROR 異常終了
   */
  condStatus init();

  /**
   * @brief 条件変数によるシグナル発行
   * @n
   * @return シグナル発行状況
   * - COND_SUCCESS 正常終了
   * - COND_ERROR 異常終了
   */
  condStatus signal();

  /**
   * @brief 条件変数によるウエイト
   * @n
   * @return ウエイト状況
   * - COND_SUCCESS 正常終了
   * - COND_ERROR 異常終了
   */
  condStatus wait();

  private:
		EventGroupHandle_t m_cond;		///< condition variable event group

    // define
    #define EVENT_FLAG_ID_1 0x00000001 ///< イベントフラグID_1
    #define EVENT_FLAG_ID_2 0x00000002 ///< イベントフラグID_2

};

#endif /* AXCCONDITIONVARIABLEFREERTOS_HPP_ */
