/**
 * @file   AXCStdioFreeRTOS.hpp
 * @brief  AXCStdioFreeRTOSクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCSTDIOFREERTOS_HPP_
#define AXCSTDIOFREERTOS_HPP_

#include "AXCStdio.hpp"

/**
 * @brief 標準入出力時機種依存定義
 */
namespace axstdio
{
  const char endl = '\r';   ///< FreeRTOS時改行コード	// baba 一旦現状は\rとしておくが実機による確認は必要
} // namespace axstdio

/**
 * @class AXCStdioFreeRTOS
 * @brief AXFStdio に対し、標準入出力機能を提供する。
 *
 * @n     [注意]AXM層は本クラスは直接使用せず、AXFStdio を使用し、OS依存しないようにする。
 * @n     使用方法
 * @n     1. インスタンスを生成する
 * @n        標準出力へログを出力するより前で宣言する必要があります
 * @n       AXFStdio stdio;
 * @n     2. 標準出力へログを出力する
 * @n        改行する場合は、std::endlではなくaxstdio::endlを使用する必要があります
 * @n        (axstdio::endlはバッファフラッシュを伴いませんので、手順３でフラッシュして出力する必要があります)
 * @n       std::cout << "出力したい文字列" << axstdio::endl;
 * @n     3. 標準出力のログをフラッシュする
 * @n       stdio.flush();
 * @n     4. 標準入力した結果を得る
 * @n       １行入力の場合
 * @n        std::string inData = stdio.getline();
 * @n       １文字入力の場合
 * @n        char inData = stdio.getchar();
 */
class AXCStdioFreeRTOS : public AXCStdio {
 public:

  /**
   * @brief AXCStdioFreeRTOSコンストラクタ
   * @n     FreeRTOS時はstd::coutへの標準出力をローカルバッファへ変更する
   */
  AXCStdioFreeRTOS();

  /**
   * @brief AXCStdioFreeRTOSデストラクタ
   * @n     FreeRTOS時は変更した標準出力バッファを元に戻す
   */
  virtual ~AXCStdioFreeRTOS();

  /**
   * @brief flush
   * @n     std::coutへ標準出力した内容を出力する
   */
  void flush();
  
  /**
   * @brief  getLine
   * @n      標準入力から１行入力する
   * @return １行分の標準入力文字列
   */
  std::string getLine();

  /**
   * @brief  getChar
   * @n      標準入力から１文字入力する
   * @return １文字分の標準入力文字
   */
  char getChar();

 private:
  std::stringbuf  mBuf;      ///< FreeRTOS時標準出力用バッファ
  std::streambuf* mPrevBuf;  ///< 使用前標準出力バッファアドレス
	bool mIsInited;
};

#endif /* AXCSTDIOFREERTOS_HPP_ */
