/**
 * @file   AXCTimerUnit.hpp
 * @brief  AXCTimerUnitクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCTIMERUNIT_HPP_
#define AXCTIMERUNIT_HPP_

#include "AXCStdint.hpp"

/**
 * @struct CoreTimeSpec
 * @brief タイマ設定に使用する構造体(秒、秒未満)
 */
struct CoreTimeSpec {
  ax::INT32_t sec;   ///< 秒
  ax::INT32_t nsec;  ///< 秒未満
};

class AXCTimerTask;

/**
 * @class AXCTimerUnit
 * @brief AXC層の TimerUnitクラス に対し、抽象クラスを提供する
 */
class AXCTimerUnit {

 public:

  /**
   * @enum
   * @brief TimerUnitメソッド実行完了状態
   */
  enum {
    TIMER_UNIT_SUCCESS = 0,
    TIMER_UNIT_ERROR_INIT = -10,
    TIMER_UNIT_ERROR_SET_ID = -20,
    TIMER_UNIT_ERROR_SET_SETTIME = -21,
    TIMER_UNIT_ERROR_SET_CRECYC = -22,
    TIMER_UNIT_ERROR_SET_NOTASK = -23,
    TIMER_UNIT_ERROR_SET_NOACT = -24,
    TIMER_UNIT_ERROR_SET_TASKADD = -25,
    TIMER_UNIT_ERROR_GET_ID = -30,
    TIMER_UNIT_ERROR_GET_GETTIME = -31,
    TIMER_UNIT_ERROR_GET_REFCYC = -32,
    TIMER_UNIT_ERROR_CANCEL_ID = -40,
    TIMER_UNIT_ERROR_CANCEL_SETTIME = -41,
    TIMER_UNIT_ERROR_CANCEL_DELCYC = -42,
    TIMER_UNIT_ERROR_GETEXPNUM_NOACT = -50,
    TIMER_UNIT_ERROR_GETEXPNUM_READ = -51,
    TIMER_UNIT_ERROR_STOP_ID = -60,
    TIMER_UNIT_ERROR_STOP_STPCYC = -61,
    TIMER_UNIT_ERROR_DELFROMTASK_NOTASK = -70,
    TIMER_UNIT_ERROR_DELFROMTASK_TASKDEL = -71,
    TIMER_UNIT_ERROR_SENDEVENT_NOTOACTOR = -80,
    TIMER_UNIT_ERROR_SENDEVENT_SEND = -81,
  };

  /**
   * @brief AXCTimerUnitコンストラクタ
   * @n
   * @param [in] task タイマタスク
   */
  AXCTimerUnit(AXCTimerTask* task);

  /**
   * @brief AXCTimerUnitデストラクタ
   */
  virtual ~AXCTimerUnit();

  /**
   * @brief create
   * @n     タイマを初期化する
   * @n
   * @return 設定可否
   */
  virtual int init() = 0;

  /**
   * @brief set
   * @n     タイマを設定、開始する
   * @n
   * @param[in]  spec          周期時間（秒、秒未満）
   * @param[in]  isSingle      単発タイマか否か
   * @n
   * @return 設定可否
   */
  virtual int set(CoreTimeSpec* spec, bool isSingle) = 0;

  /**
   * @brief get
   * @n     タイマ情報を取得する
   * @n
   * @param[out] *ret_spec タイマ周期(満了)の経過時間（秒、秒未満）
   * @n
   * @return 設定可否
   */
  virtual int get(CoreTimeSpec* ret_spec) = 0;

  /**
   * @brief cancel
   * @n     タイマをキャンセルする
   * @n
   * @return 設定可否
   */
  virtual int cancel() = 0;

  /**
   * @brief getExpNum
   * @n     本メソッドの前回呼出しが返ってからの
   * @n     タイマの満了回数を返す
   * @n
   * @return タイマの満了回数
   */
  virtual int getExpNum() = 0;
};

#endif /* AXCTIMERUNIT_HPP_ */
