/**
 * @file   AXFCommInternal.hpp
 * @brief  AXFCommInternalクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFCOMMINTERNAL_HPP_
#define AXFCOMMINTERNAL_HPP_

#include "AXCCommFreeRTOS.hpp"
#include "AXFEvent.hpp"

/**
 * @class AXFCommInternal
 * @brief AXF送受信スレッドクラス(AXFCommSendInternal/AXFCommReceiveInternal)に対し、
 *         共有メモリの同期write/read機能を提供する
 *
 * @n     AXFCommInternalクラスの 詳細は AXCCommクラス の説明に記述する。
 *
 */
class AXFCommInternal :
	public AXCCommFreeRTOS
{
 public:
  // GoogleTest対応のため、仮想関数として実装する */
  /**
   * @brief 初期化
   * @param [in] type AXCommタイプ
   * @return AXComm初期化状況
   * - COMM_SUCCESS 正常終了
   * - COMM_ERROR 異常終了
   */
  virtual commStatus init(AXComm_t type);

  /**
   * @brief 同期読み出し
   * @param [out] buf
   * @param [in] size
   * @return 読み出しデータサイズ(エラー時は 0 未満)
   */
  virtual int readSync(char *buf, int size);

  /**
   * @brief 同期書き込み
   * @param [in] buf
   * @param [in] size
   * @return 書き込みデータサイズ(エラー時は 0 未満)
   */
  virtual int writeSync(char *buf, int size);

  /**
   * @brief      宛先NameObject文字列ポインタの取得
   * @n
   * @param[in]  buf         受信データバッファ
   * @n
   * @return NameObject文字列ポインタ
   */
  virtual char* getObjectNameStringPtr(char* buf);

  /**
   * @brief      受信イベントIDの取得
   * @n
   * @param[in]  buf         受信データバッファ
   * @n
   * @return イベントID
   */
  virtual AXFEvent_t getEventId(char* buf);

  /**
   * @brief      受信データサイズの取得
   * @n
   * @param[in]  buf         受信データバッファ
   * @n
   * @return データサイズ
   */
  virtual INT32_t getDataSize(char* buf);

  /**
   * @brief      受信データポインタの取得
   * @n
   * @param[in]  buf         受信データバッファ
   * @n
   * @return データポインタ
   */
  virtual char* getDataPtr(char* buf);

  /**
   * @brief      送信データの設定
   * @n
   * @param[in]  buf         送信データバッファ
   * @param[in]  bufSize     送信データバッファサイズ
   * @param[in]  name        送信宛Name
   * @param[in]  eventId     送信イベントID
   * @param[in]  data        送信データ
   * @param[in]  size        送信データサイズ
   * @n
   * @return 設定データサイズ (失敗時は 0 未満の値を返す)
   */
  virtual INT32_t setPacket(char* buf, INT32_t bufSize, const char* name,
                            AXFEvent_t eventId, void* data, INT32_t size);
};

#endif /* AXFCOMMINTERNAL_HPP_ */
