/**
 * @file   AXFTimerInternal.hpp
 * @brief  AXFTimerInternalクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFTIMERINTERNAL_HPP_
#define AXFTIMERINTERNAL_HPP_

#include "AXFTimerUnitInternal.hpp"
#include "AXFTimerTaskInternal.hpp"

#include "AXFActor.hpp"
#include "AXFObjectName.hpp"

/**
 * @class AXFTimerInternal
 * @brief  AXFActorクラスに対し、タイマ登録・削除機能を提供する
 *
 */
class AXFTimerInternal
{
  AXFObjectName& m_name;

 public:

  /**
   * @enum  TimerKind
   * @brief Timerの種別
   */
  enum TimerKind {
    SINGLE_SHOT_TIMER = 0,  ///< 単発タイマ
    INTERVAL_TIMER          ///< 周期タイマ
  };

 public:
  /**
   * @brief AXFTimerInternalコンストラクタ
   * @n
   * @param [in] name タイマを使用するアクタ名
   */
  AXFTimerInternal(AXFObjectName& name);

  /**
   * @brief AXFTimerInternalデストラクタ
   */
  virtual ~AXFTimerInternal();

  /**
   * @brief addTimer
   * @n     タイマの設定と開始を行う
   * @n
   * @param[in]  *obj          アクタのハンドル
   * @param[in]  timer         周期時間（sec 秒 / tick チックタイム）
   * @param[in]  kind          AXFTimerの種別
   * @param[in]  timerEventId  イベントID（時間満了時に、発行されるイベントID）
   * @param[out] *TimerId      AXFタイマID（delTimer、getTimer利用時に利用）
   * @n
   * @return 設定可否
   */
  virtual AXFActor::timerStatus addTimer(AXFActor* obj, AXFActor::timeSpec timer,
                                 TimerKind kind, AXFEvent_t timerEventId,
                                 AXFActor::TimerId_t* TimerId);
  /**
   * @brief delTimer
   * @n     タイマの停止を行う
   * @n
   * @param[in] TimerId  AXFタイマID
   * @n
   * @return 設定可否
   */
  virtual AXFActor::timerStatus delTimer(AXFActor::TimerId_t TimerId);

  /**
   * @brief getTimer
   * @n     タイマ情報の取得を行う
   * @n
   * @param[in]  TimerId    AXFタイマID
   * @param[out] *timerspec  タイマ周期(満了)の経過時間（秒 + tick）
   * @n
   * @return 設定可否
   */
  virtual AXFActor::timerStatus getTimer(AXFActor::TimerId_t TimerId,
                                 AXFActor::timeSpec* timerspec);

  /**
   * @brief initTimer
   * @n     タイマ使用前初期処理を行う
   * @n
   * @return 初期化結果
   */
  virtual AXFActor::timerStatus initTimer();

  /**
   * @brief killTimer
   * @n     タイマ情報の管理テーブルが空でない場合、タイマを停止し情報の削除を行う
   */
  virtual void killTimer();

  static void maskTimerSignal() {
  }

 private:

  // インスタンスのコピーは想定しない(Klockwork指摘対応:CL.FFM.ASSIGN)
  AXFTimerInternal& operator=(const AXFTimerInternal&) {
    return *this;
  }

  // インスタンスのコピーは想定しない(Klockwork指摘対応:CL.FFM.COPY)
  AXFTimerInternal(const AXFTimerInternal& obj)
      : m_name(obj.m_name) {
  }

 protected:
  virtual AXFTimerUnitInternal* newTimerUnit(AXFTimerTaskInternal* task, AXFObjectName& name, AXFActor* toActor, AXFEvent_t event) {
    return new (std::nothrow) AXFTimerUnitInternal(task, name, toActor, event);
  }

};

#endif /* AXFTIMERINTERNAL_HPP_ */
