/**
 * @file   AXFTimerUnitInternal.hpp
 * @brief  AXFTimerUnitInternelクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFTIMERUNITINTERNAL_HPP_
#define AXFTIMERUNITINTERNAL_HPP_

#include "AXCTimerUnitFreeRTOS.hpp"
#define AXFTIMERUNIT_BASE_CLASS AXCTimerUnitFreeRTOS

#include "AXFActor.hpp"
#include "AXFEvent.hpp"

class AXFTimerTaskInternal;

/**
 * @class AXFTimerUnitInternal
 * @brief 1つのタイマ・インスタンスを管理する機能を提供する
 *
 */
class AXFTimerUnitInternal : public AXFTIMERUNIT_BASE_CLASS {
  int m_id;

  AXFTimerTaskInternal* m_task;
  bool m_isSingle;
  AXFObjectName& m_name;
  AXFActor* m_toActor;
  AXFEvent_t m_event;

#ifndef	UNUSE_CHRONO	// baba Chrono非対応
  AXFChrono m_chronoSpec;
  AXFChrono m_chronoStart;
  AXFChrono m_chronoUpdate;
#endif	/* UNUSE_CHRONO */

 public:

  /**
   * @brief AXFTimerUnitInternalコンストラクタ
   * @n
   * @n     タイマを使用するアクタ名は、
   * @n     タイマクラスの名前指定による削除メソッドやキャンセルメソッドから、
   * @n     本クラスの getName() メソッドが呼ばれ参照される
   * @n
   * @param [in] task タイマタスク
   * @param [in] name タイマを使用するアクタ名
   * @param [in] toActor 満了時にイベントを送信する先のアクタ
   * @param [in] event 満了時に送信するイベント
   */
  AXFTimerUnitInternal(AXFTimerTaskInternal* task, AXFObjectName& name, AXFActor* toActor, AXFEvent_t event);

  /**
   * @brief AXFTimerUnitInternalデストラクタ
   */
  virtual ~AXFTimerUnitInternal();

  /**
   * @brief set
   * @n     タイマを設定、開始し、タイマタスクの登録リストに追加する
   * @n
   * @n     単発タイマを指定した場合、タイマが満了しイベント送信後に、
   * @n     本オブジェクトのインスタンスは、
   * @n     タイマタスクによって破棄される
   * @n
   * @param[in]  spec          周期時間（秒、秒未満）
   * @param[in]  isSingle      単発タイマか否か
   * @n
   * @return 設定可否
   */
  virtual int set(CoreTimeSpec* spec, bool isSingle);

  /**
   * @brief getExpNum
   * @n     本メソッドの前回呼出しが返ってからの
   * @n     タイマの満了回数を返す
   * @n
   * @return タイマの満了回数
   */
  virtual int getExpNum();

  /**
   * @brief getCorrExpNum
   * @n     タイマの満了回数を補正して返す
   * @n     TimeDomain無効に切り替わった後、
   * @n     初回の getExpNum() 呼出で取得した満了回数をを補正する
   * @n
   * @param[in]  num  タイマの満了回数
   * @n
   * @return 補正後のタイマの満了回数 (ただし負の値の場合はエラー)
   */
  virtual int getCorrExpNum(int num);

  /**
   * @brief delFromTask
   * @n     自身をタイマタスクの登録リストから削除する
   * @n
   * @return 設定可否
   */
  virtual int delFromTask();

  int sendEvent(); // AXFTimerTaskInternalクラスからのみ呼び出される
  bool isSingle(); // AXFTimerTaskInternalクラスからのみ呼び出される
  AXFObjectName& getName(); // AXFTimerTaskInternalクラスからのみ呼び出される
  void setId(int id); // AXFTimerTaskInternalクラスからのみ呼び出される
  int getId(); // AXFTimerTaskInternalクラスとAXFTimerInternalクラスから呼び出される

  AXFActor* getToActor();
  bool isNeverSimu();
  bool isTimeDomainActive();

 private:

#ifndef	UNUSE_CHRONO	// baba Chrono非対応
  int getExpNumChrono(AXFChrono now);
  int getExpNumChrono(AXFChrono from, AXFChrono to);
#endif	/* UNUSE_CHRONO */
};

#endif /* AXFTIMERUNITINTERNAL_HPP_ */
