/**
 * @file   AXHDefect.hpp
 * @brief  Header file of Defect class.
 * @par    Programming Language
 *         C++
 */

#ifndef A_X_H_DEFECT_H
#define A_X_H_DEFECT_H

#include "AXHSpecParam.hpp"
#include "AXHSpecDefectStruct.hpp"

/**
 * @class AXHDefecf
 * @brief Defect class
 * 
 */
class AXHDefect {
 private:
  /**
   * @brief アドレス指定モード設定値
   */
  unsigned char DMAmode;
  /**
   * @brief 入力画像高さ設定値
   */
  unsigned char AutoEnable;
  /**
   * @brief 補正画素アドレス/補正方向設定値
   */
  ST_DEFECT_ADDRESS PixcelAddr[AXH_Defect_IntAddr_Max];
  /**
   * @brief 自動補正 補正方法
   */
  unsigned char AutoCorrect;
  /**
   * @brief 自動補正 参照画素選択
   */
  unsigned char AutoSel;
  /**
   * @brief 白点画素検出用しきい値設定値
   */
  unsigned short WhiteDefectTH;
  /**
   * @brief 黒点画素検出用しきい値設定値
   */
  unsigned short BlackDefectTH;
  /**
   * @brief 黒点画素検出用しきい値設定値
   */
  unsigned short LevelModulate[AXH_Defect_ADFOFS_Max];
  /**
   * @brief 画素欠陥補正データ参照先
   */
  unsigned int *p_RegSPU_DMAR1_OFS;
  /**
   * @brief SPUコントロールレジスタ参照先
   */
  unsigned int *p_RegSPU_C;
  /**
   * @brief 欠陥補正制御レジスタ参照先
   */
  unsigned int *p_RegDEFCT_C;
  /**
   * @brief 欠陥画素設定レジスタ参照先
   */
  unsigned int *p_RegCDEF[AXH_Defect_IntAddr_Max];
  /**
   * @brief 白点画素検出しきい値設定参照先
   */
  unsigned int *p_RegADFCTTH_H;
  /**
   * @brief 黒点画素検出しきい値設定参照先
   */
  unsigned int *p_RegADFCTTH_L;
  /**
   * @brief 輝度信号レベル変調設定参照先
   */
  unsigned int *p_RegADFOFS[AXH_Defect_ADFOFS_REG_Max];

public:
  /**
   * @brief  オブジェクトを初期化する
   */
  AXHDefect();

  /**
   * @brief  オブジェクトを開放する
   */
  ~AXHDefect();

  /**
   * @brief AXHOpdConfigBaseクラスを初期化する
   */
  void Init();

  /**
   * @brief アドレス指定モード(ON/OFF)を設定する
   *
   * @param [in] mode ON/OFF設定
   */
  void setDMAmode(unsigned char mode);

  /**
   * @brief アドレス指定モード(ON/OFF)の設定値を取得する
   *
   * @return ON/OFF設定
   */
  unsigned char getDMAmode();

  /**
   * @brief 自動補正(ON/OFF)を設定する
   *
   * @param [in] mode ON/OFF設定
   */
  void setAutoEnable(unsigned char mode);

  /**
   * @brief 自動補正(ON/OFF)の設定値を取得する
   *
   * @return ON/OFF設定
   */
  unsigned char getAutoEnable();

  /**
   * @brief 欠陥画素アドレスを設定する
   *
   * @param [in] index インデックス番号
   * @param [in] addr 水平方向座標/垂直方向座標/補正方法
   */
  void setPixcelAddr(unsigned int index, ST_DEFECT_ADDRESS addr);

  /**
   * @brief 欠陥画素アドレスの設定を取得する
   *
   * @param [in]  index インデックス番号
   * @return 水平方向座標/垂直方向座標/補正方法
   */
  ST_DEFECT_ADDRESS getPixcelAddr(unsigned int index);

  /**
   * @brief 自動補正機能の補正方法を設定する
   *
   * @param [in] mode 補正方法
   */
  void setAutoCorrect(unsigned char mode);

  /**
   * @brief 自動補正機能の補正方法の設定を取得する
   *
   * @return 補正方法
   */
  unsigned char getAutoCorrect();

  /**
   * @brief 自動補正の平均化の際の参照画素を設定する
   *
   * @param [in] sel AVEモード
   */
  void setAutoSel(unsigned char sel);

  /**
   * @brief 自動補正の平均化の際の参照画素の設定を取得する
   *
   * @return AVEモード
   */
  unsigned char getAutoSel();

  /**
   * @brief 白点検出用しきい値を設定する
   *
   * @param [in] th 白点検出用しきい値
   */
  void setWhiteDefectTH(unsigned short th);

  /**
   * @brief 白点検出用しきい値の設定を取得する
   *
   * @return 白点検出用しきい値
   */
  unsigned short getWhiteDefectTH();

  /**
   * @brief 垂直分割画素数を設定する
   *
   * @param [in] th 黒点検出用しきい値
   */
  void setBlackDefectTH(unsigned short th);

  /**
   * @brief 垂直分割画素数の設定を取得する
   *
   * @return 黒点検出用しきい値
   */
  unsigned short getBlackDefectTH();

  /**
   * @brief 輝度信号レベルを設定する
   *
   * @param [in] index インデックス番号
   * @param [in] level 輝度信号レベル
   */
  void setLevelModulate(unsigned int index, unsigned short level);

  /**
   * @brief 輝度信号レベルの設定を取得する
   *
   * @param [in] index インデックス番号
   * @return 黒点検出用しきい値
   */
  unsigned short getLevelModulate(unsigned int index);

};
#endif
