/**
 * @file   AXHMatrixGain.hpp
 * @brief  Header file of Matrix Gain Class
 * @par    Programming Language
 *         C++
 */
#ifndef A_X_H_MATRIX_GAIN_H
#define A_X_H_MATRIX_GAIN_H

#include "AXHCommonParam.hpp"
#include "AXHSpecParam.hpp"

/**
 * @class AXHMatrixGain
 * @brief Matrix Gain Class
 * 
 */
class AXHMatrixGain {
 private:
  /**
   * @brief カラーマトリックスモード設定値
   */
  unsigned int mode;
  /**
   * @brief RSZ_C0レジスタ参照先
   */
  unsigned int *p_c0;
  /**
   * @brief Crクロマゲイン設定値
   */
  unsigned int crGain[AXHMatrixGainMax];
  /**
   * @brief Crクロマゲイン設定レジスタ参照先
   */
  unsigned int *p_crGain[AXHMatrixGainMax];
  /**
   * @brief Cbクロマゲイン設定値
   */
  unsigned int cbGain[AXHMatrixGainMax];
  /**
   * @brief Cbクロマゲイン設定レジスタ参照先
   */
  unsigned int *p_cbGain[AXHMatrixGainMax];
  /**
   * @brief Crマトリックスゲイン設定値
   */
  unsigned int crcbGain[AXHMatrixGainMax];
  /**
   * @brief Crマトリックスゲイン設定レジスタ参照先
   */
  unsigned int *p_crMatrixGain[AXHMatrixGainMax];
  /**
   * @brief Cbマトリックスゲイン設定値
   */
  unsigned int cbcrGain[AXHMatrixGainMax];
  /**
   * @brief Cbマトリックスゲイン設定レジスタ参照先
   */
  unsigned int *p_cbMatrixGain[AXHMatrixGainMax];
  /**
   * @brief Y-Gain設定値
   */
  unsigned int YGain;
  /**
   * @brief CSCFゲイン(Yゲイン)設定レジスタ参照先
   */
  unsigned int *p_YGain[AXHMatrixGainMax];
  /**
   * @brief Y-Offset設定値
   */
  unsigned short YOffset;
  /**
   * @brief CSCオフセット設定レジスタ参照先
   */
  unsigned int *p_CSOS[AXHMatrixOffsetMax];
  /**
   * @brief CSCビットコントロールレジスタ参照先
   */
  unsigned int *p_CSBC;

 public:
  /**
   * @brief オブジェクトを初期化する
   */
  AXHMatrixGain();

  /**
   * @brief オブジェクトを開放する
   */
  ~AXHMatrixGain();

  /**
   * @brief カラーマトリックスモード設定値を取得する
   *
   * @return カラーマトリックスモード設定値
   */
  unsigned int getMode();

  /**
   * @brief カラーマトリックスモードを設定する
   *
   * @param [in] mode カラーマトリックスモード設定値
   */
  void setMode(unsigned int mode);

  /**
   * @brief Crクロマゲイン設定値を取得する
   *
   * @param [in] index 象限番号
   *
   * @return Crクロマゲイン設定値
   */
  unsigned int getCrGain(unsigned int index);

  /**
   * @brief Crクロマゲインを設定する
   *
   * @param [in] index 象限番号
   * @param [in] crGain Crクロマゲイン
   */
  void setCrGain(unsigned int index, unsigned int crGain);

  /**
   * @brief Cbクロマゲイン設定値を取得する
   *
   * @param [in] index 象限番号
   *
   * @return Cbクロマゲイン設定値
   */
  unsigned int getCbGain(unsigned int index);

  /**
   * @brief Cbクロマゲインを設定する
   *
   * @param [in] index 象限番号
   * @param [in] cbGain Cbクロマゲイン
   */
  void setCbGain(unsigned int index, unsigned int cbGain);

  /**
   * @brief crマトリックスゲインを取得する
   *
   * @param [in] index 象限番号
   *
   * @return Crマトリックスゲイン設定値
   */
  unsigned int getCrCbGain(unsigned int index);

  /**
   * @brief Crマトリックスゲインを設定する
   *
   * @param [in] index 象限番号
   * @param [in] crcbGain Crマトリックスゲイン
   */
  void setCrCbGain(unsigned int index, unsigned int crcbGain);

  /**
   * @brief Cbマトリックスゲインを設定する
   *
   * @param [in] index 象限番号
   *
   * @return Cbマトリックスゲイン設定値
   */
  unsigned int getCbCrGain(unsigned int index);

  /**
   * @brief Cbマトリックスゲインを設定する
   *
   * @param [in] index 象限番号
   * @param [in] cbcrGain Cbマトリックスゲイン
   */
  void setCbCrGain(unsigned int index, unsigned int cbcrGain);
  /**
   * @brief Y-Gainを設定する
   *
   * @return Y-Gain設定値
   */
  unsigned int getYGain();

  /**
   * @brief Y-Gainを設定する
   *
   * @param [in] yGain Y-Gain
   */
  void setYGain(unsigned int yGain);
  /**
   * @brief Y-Offsetを設定する
   *
   * @return Y-Offset設定値
   */
  unsigned short getYOffset();

  /**
   * @brief Y-Offsetを設定する
   *
   * @param [in] yOffset Y-Offset
   */
  void setYOffset(unsigned short yOffset);
};


/**
 * @class AXHRGBMatrixCoef
 * @brief RGB Matrix Coef Class
 *
 */
class AXHRGBMatrixCoef {
 private:
  /**
   * @brief マトリックス係数設定値
   */
  unsigned short Coef[AXH_RGBMatrixOut_Max][AXH_RGBMatrixIn_Max];
  /**
   * @brief マトリックス設定レジスタ参照先
   */
  unsigned int *p_Coef[AXH_RGBMatrixOut_Max][AXH_RGBMatrixIn_Max];
  /**
   * @brief マトリックスオフセット設定値
   */
  unsigned short Offset[AXH_RGBMatrixOut_Max];
  /**
   * @brief マトリックスオフセット設定レジスタ参照先
   */
  unsigned int *p_Offset[AXH_RGBMatrixOut_Max];
  /**
   * @brief マトリックス制御設定値
   */
  unsigned int Ctrl;
  /**
   * @brief マトリックス制御レジスタ参照先
   */
  unsigned int *p_Ctrl;

 public:
  /**
   * @brief  AXHRGBMatrixCoefクラスコンストラクタ
   *
   * @n      オブジェクトを初期化する
   */
  AXHRGBMatrixCoef();

  /**
   * @brief  AXHRGBMatrixCoefクラスデストラクタ
   *
   * @n      オブジェクトを開放する
   */
  ~AXHRGBMatrixCoef();

  /**
   * @brief カラーマトリックスモード設定値を取得する
   *
   * @param [in] inIndex 入力ポジション
   * @param [in] outIndex 出力ポジション
   * @return カラーマトリックスモード設定値
   */
  unsigned short getCoef(unsigned int inIndex, unsigned int outIndex );

  /**
   * @brief カラーマトリックスモードを設定する
   *
   * @param [in] inIndex 入力ポジション
   * @param [in] outIndex 出力ポジション
   * @param [in] coef マトリックス係数
   */
  void setCoef(unsigned int inIndex, unsigned int outIndex, unsigned short coef);

  /**
   * @brief マトリックスオフセット設定値を取得する
   *
   * @param [in] outIndex 出力ポジション
   * @return マトリックスオフセット設定値
   */
  unsigned short getOffset(unsigned int outIndex);

  /**
   * @brief マトリックスオフセットを設定する
   *
   * @param [in] index 出力ポジション
   * @param [in] offset マトリックスオフセット
   */
  void setOffset(unsigned int index, unsigned short offset);

  /**
   * @brief 制御レジスタ設定値を取得する
   *
   * @return 制御レジスタ設定値
   */
  unsigned int getRegCtrl();

  /**
   * @brief 制御レジスタを設定する
   *
   * @param [in] ctrl 制御レジスタ設定値
   */
  void setRegCtrl(unsigned int ctrl);
  };

#endif
