/**
 * @file			AXMIspGammaSettings.hpp
 * @brief			AXMIspGammaSettingsクラスヘッダ
 *
 * @par			言語
 * 					C++
 */

#ifndef AXMISPGAMMASETTINGS_HPP_
#define AXMISPGAMMASETTINGS_HPP_

#include "AXMIspAdjustType.hpp"
#include "../AXMCommon/AXMCommonStruct.hpp"
/**
 * @enum  		GammaMode
 * @brief		ガンマ補正モード
 */
typedef enum {
	GammaPreset,				//!< プリセットモード
	GammaUserset,				//!< ユーザー設定モード
} GammaMode;

// ガンマモード数
const BYTE				GAMMA_MODE_MAX = GammaUserset + 1;

/**
 * @enum  		GammaCurve
 * @brief		ガンマ補正カーブ選択
 */
typedef enum {
	GammaCurve_1,				//!< γ1
	GammaCurve_2,				//!< γ1.2
	GammaCurve_3,				//!< γ1.4
	GammaCurve_4,				//!< γ1.6
	GammaCurve_5,				//!< γ1.8
	GammaCurve_6,				//!< γ2
	GammaCurve_7,				//!< γ2.2
	GammaCurve_8,				//!< γ2.4
	GammaCurve_9,				//!< γ2.6
} GammaCurve;

// ガンマ補正カーブ選択数
const BYTE				GAMMA_CURVE_MAX = GammaCurve_9 + 1;

// ガンマ補正値テーブル(x10値)
const SHORT			GammaCurveTable[GAMMA_CURVE_MAX] = { 10, 12, 14, 16, 18, 20, 22, 24, 26 };

// ガンマ補正暗部補正頂点数
const BYTE				GAMMA_DARK_CORRECT_COUNT = 6;

// ガンマ補正暗部補正テーブル(最初の６点のみに適応)
const UINT				GAMMA_DARK_CORRECT_TABLE[GAMMA_CURVE_MAX][GAMMA_DARK_CORRECT_COUNT] = {
	  { 0,   0,   0,   0,   0,   0 }  // 1.0
	, { 0,  33,  21,  11,   5,   2 }  // 1.2
	, { 0,  67,  42,  22,  10,   3 }  // 1.4
	, { 0, 100,  63,  33,  15,   5 }  // 1.6
	, { 0, 133,  83,  43,  20,   7 }  // 1.8
	, { 0, 167, 104,  54,  25,   8 }  // 2.0
	, { 0, 200, 125,  65,  30,  10 }  // 2.2
	, { 0, 233, 146,  76,  35,  12 }  // 2.4
	, { 0, 267, 167,  87,  40,  13 }  // 2.6
};

/**
 * @enum  		GammaComp
 * @brief		ガンマカーブ圧縮設定
 */
typedef enum {
	GammaComp_1,				//!< 1段階
	GammaComp_2,				//!< 2段階
	GammaComp_3,				//!< 3段階
	GammaComp_4,				//!< 4段階
	GammaComp_5,				//!< 5段階
	GammaComp_6,				//!< 6段階
	GammaComp_7,				//!< 7段階
	GammaComp_8,				//!< 8段階
} GammaComp;

// ガンマカーブ圧縮設定最大数
const BYTE				GAMMA_COMP_MAX = GammaComp_8 + 1;

const SHORT			DEF_KNEE_X = 500;						//!< KneeポイントXデフォルト値(x1000値)
const SHORT			DEF_KNEE_Y = 850;						//!< KneeポイントYデフォルト値(x1000値)
const SHORT			DEF_KNEE_MAX = 1000;
const SHORT			PIXEL_MAX  = 0x1000;					//!< 画素値最大(4096)

/**
 * @class		KneeSettings
 * @brief		ガンマ補正Knee設定クラス
 */
class KneeSettings {
public:
	/**
	 * @brief  オブジェクトを初期化する
	 *
	 * @n
	 */
	KneeSettings();
	/**
	 * @brief  オブジェクトを開放する
	 *
	 * @n
	 */
	~KneeSettings();
public:
	SHORT				dRange;								//!< Dレンジ設定
	SHORT				hRange;								//!< ハイライト領域設定
};

const short			CURVE_POI_CNT = 65;					//!< ガンマ補正カーブ頂点数

/**
 * @class     AXMIspGammaSettings
 * @brief     ガンマ補正設定パラメータクラス
 *
 * @n         ガンマ補正の計算パラメータと、計算結果を格納する
 */
class AXMIspGammaSettings {
public:
	/**
	 * @brief  オブジェクトを初期化する
	 *
	 * @n
	 */
	AXMIspGammaSettings();
	/**
	 * @brief  オブジェクトを開放する
	 *
	 * @n
	 */
	virtual ~AXMIspGammaSettings();
public:
	GammaMode			gammaMode;								//!< モード設定
	GammaCurve			gammaCurveSelect;						//!< ガンマ補正カーブ選択
	KneeSettings		kneeSettings;							//!< Knee設定
	USHORT				defGammaCurvePoint[CURVE_POI_CNT];	//!< デフォルトガンマ補正カーブ頂点座標
	USHORT				usrGammaCurvePoint[CURVE_POI_CNT];	//!< ユーザーガンマ補正カーブ頂点座標
	USHORT				setGammaCurvePoint[CURVE_POI_CNT];	//!< 出力ガンマ補正カーブ頂点座標
	GammaComp			gammaCompLevelSelect;				//!< 圧縮レベル設定
};

#endif /* AXMISPGAMMASETTINGS_HPP_ */
