/**
 * @file			AXMIspObSettings.hpp
 * @brief			AXMIspObSettingsクラスヘッダ
 *
 * @par			言語
 * 					C++
 */

#ifndef AXMISPOBSETTINGS_HPP_
#define AXMISPOBSETTINGS_HPP_

#include "AXMIspAssistType.hpp"

/**
 * @enum  		OBMode
 * @brief		OB調整機能モード
 */
typedef enum {
	OBFeedback,				//!< Feedbackモード
	OBFixvalue,				//!< 固定値モード
} OBMode;

/**
 * @enum  		OBSensorArrayPatternSelect
 * @brief		センサー配列パターン選択
 */
typedef enum OBSensorArrayPatternSelect {
	OBArrayPattern_2_2,		//!< 2x2
	OBArrayPattern_2_4,		//!< 2x4
	OBArrayPattern_4_2,		//!< 4x2
	OBArrayPattern_4_4,		//!< 4x4
	OBArrayPattern_2_8,		//!< 2x8
	OBArrayPattern_8_2,		//!< 8x2
} OBArrayPattern;

/**
 * @class		AXMIspOBSamplingRange
 * @brief		OBサンプリング領域設定クラス
 *
 * @n			OBのサンプリング領域設定パラメータを格納する\n
 * @n			OBフィードバックモード時に有効
 */
typedef class AXMIspOBSamplingRange {
public:
	/**
	 * @brief  オブジェクトを初期化する
	 *
	 * @n
	 */
	AXMIspOBSamplingRange();
	/**
	 * @brief  オブジェクトを開放する
	 *
	 * @n
	 */
	virtual ~AXMIspOBSamplingRange();
public:
	/**
	 * @brief	SetHorizon
	 * @n		OBサンプリング水平タイミングの開始と終了を設定する
	 *
	 * @param	[in]  start                    開始
	 * @param	[in]  end                      終了
	 */
	void 				SetHorizon(SHORT start, SHORT end);
	/**
	 * @brief	SetVertical
	 * @n		OBサンプリング垂直タイミングの開始と終了を設定する
	 *
	 * @param	[in]  start                    開始
	 * @param	[in]  end                      終了
	 */
	void 				SetVertical(SHORT start, SHORT end);
	/**
	 * @brief	GetWidth
	 * @n		OBサンプリングの画像幅取得
	 *
	 * @return	OBサンプリングの画像幅
	 */
	SHORT				GetWidth();
	/**
	 * @brief	GetHeight
	 * @n		OBサンプリングの画像高さ取得
	 *
	 * @return	OBサンプリングの画像高さ
	 */
	SHORT				GetHeight();
public:
	SHORT				hStart;								//!< OBサンプリング開始水平タイミング
	SHORT				vStart;								//!< OBサンプリング開始垂直タイミング
	SHORT				hEnd;									//!< OBサンプリング終了水平タイミング
	SHORT				vEnd;									//!< OBサンプリング終了垂直タイミング
} OBSamplingRange;

const int				OB_PTN_PIXEL_CNT = 16;				//!< センサー配列パターンのピクセル最大数
const int				OB_DRK_VALUE_CNT = 8;				//!< 黒レベル設定の減算値最大数
const int				OB_ACCUM_CNT  = 4;					//!< 積算カウンタ数(R/Gr/B/Gb)

/**
 * @class		AXMIspObSettings
 * @brief		OB調整設定パラメータクラス
 *
 * @n         OB調整の計算パラメータと、計算結果を格納する
 */
class AXMIspObSettings {
public:
	AXMIspObSettings();
	virtual ~AXMIspObSettings();
public:
	OBMode				obMode;                             //!< モード設定
	OBSamplingRange	samplingRange;                      //!< OBサンプリング領域
	OBArrayPattern	arrayPattern;                       //!< センサー配列パターン選択
	BYTE				arrayPtnIdx     [OB_PTN_PIXEL_CNT]; //!< 黒レベル設定減算値のインデックス(センサー配列パターンの各ピクセル持つ)
	UINT				indexValue      [OB_DRK_VALUE_CNT]; //!< 黒レベル設定減算値
	USHORT				obDarkLevelFloor[OB_ACCUM_CNT];     //!< OB黒レベル床値[0:R(左上)/1:Gr(右上)/2:B(右下)/3:Gb(左下)]
	UINT				obAccumlator    [OB_ACCUM_CNT];     //!< OB積算値[0:R(左上)/1:Gr(右上)/2:B(右下)/3:Gb(左下)]
	UINT				obDarkLevel     [OB_ACCUM_CNT];     //!< OB黒レベル[0:R(左上)/1:Gr(右上)/2:B(右下)/3:Gb(左下)]
};

#endif /* AXMISPOBSETTINGS_HPP_ */
