/**
 * @file   AXMPixcessorState.hpp
 * @brief  AXMPixcessorクラス StateMachineオブジェクトヘッダ
 *
 * @n      Pixcessor IFクラス
 *         Across ISP
 * @par    Programming Language
 *         C++
 */

  /**
   * @enum  pixcessorModelState
   * @brief StateMachine用状態定義
   */
  enum pixcessorModelState {
    STATUS_RUNNING = 0,                 //!<  動作中
    STATSU_MAX
  };

  /**
   * @enum  pixcessorModelEvent
   * @brief StateMachine用イベント定義
   */
  enum pixcessorModelEvent {
    PIXCESSOR_EVENT_FAKE_ISR = AXFEVENT_MODEL(0),                               //!<  AXシミュレータ用擬似フレーム完了割り込み
#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
    ISP_EVENT_AWB_OPD_SET,                                                      //!<  AWB OPD設定要求
    ISP_EVENT_AE_OPD_SET,                                                       //!<  AE OPD設定要求
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
#ifndef ENABLE_OPD_DATA_ISP_TO_AXH
    ISP_EVENT_AWB_IMAGE,                                                        //!<  AWB 画像データ取得要求
    ISP_EVENT_AE_IMAGE,                                                         //!<  AE 画像データ取得要求
#endif  //  ENABLE_OPD_DATA_ISP_TO_AXH
#ifndef ENABLE_IMAGE_ADDR_ISP_TO_AXH
    ISP_EVENT_SHADE_IMAGE,                                                      //!<  Correct(シェーディング補正) 画像データ取得要求
#endif  //  ENABLE_IMAGE_ADDR_ISP_TO_AXH
    ISP_EVENT_OB_AREA_SET,                                                      //!<  OB OB領域設定要求
    ISP_EVENT_OB_DARKLEVEL_SET,                                                 //!<  OB 黒レベル設定要求
    ISP_EVENT_OB_ACCUM_GET,                                                     //!<  OB OB積算データ取得要求
    ISP_EVENT_OB_SENSOR_ARRAY_PTN_SET,                                          //!<  OB センサ配列パターン選択領域設定要求
    ISP_EVENT_OB_ARRAY_PTN_INDEX_SET,                                           //!<  OB 配列パターンへのインデックス設定値設定要求
    ISP_EVENT_OB_FIX_DARKLEVEL_SET,                                             //!<  OB 固定黒レベル値設定要求
    ISP_EVENT_GAMMA_TABLE_SET,                                                  //!<  Gamma Gammaテーブル設定要求
    ISP_EVENT_APERTURE_TOTAL_GAIN_SET,                                          //!<  Aperture トータルゲイン設定要求
    ISP_EVENT_APERTURE_LIMIT_SET,                                               //!<  Aperture リミット設定要求
    ISP_EVENT_APERTURE_OVERMIDRANGE_GAINRATIO_SET,                              //!<  Aperture 高域・中域ゲイン比率設定要求
    ISP_EVENT_APERTURE_LUT_TABLE_SET,                                           //!<  Aperture LUTテーブル設定要求
    ISP_EVENT_APERTURE_CORING_SET,                                              //!<  Aperture コアリング設定要求
    PIXCESSOR_EVENT_LENS_SHADE_SET,                                             //!<  レンズシェーディング補正 設定要求
    PIXCESSOR_EVENT_IRIDIX_SET,                                                 //!<  IRIDIX 設定要求
    PIXCESSOR_EVENT_NR_SET,                                                     //!<  NR 一括設定要求
    PIXCESSOR_EVENT_NR_SIZE_SET,                                                //!<  NR Size設定要求
    PIXCESSOR_EVENT_NR_MODE_SET,                                                //!<  NR Mode設定要求
    PIXCESSOR_EVENT_NR_MAINY_SET,                                               //!<  NR MainY設定要求
    PIXCESSOR_EVENT_NR_MAINC_SET,                                               //!<  NR MainC設定要求
    PIXCESSOR_EVENT_NR_SUBY_SET,                                                //!<  NR SubY設定要求
    PIXCESSOR_EVENT_NR_SUBC_SET,                                                //!<  NR SubC設定要求
    PIXCESSOR_EVENT_DEFECT_SET,                                                 //!<  画素欠陥補正 一括設定要求
    PIXCESSOR_EVENT_DEFECT_MODE_SET,                                            //!<  画素欠陥補正 モード設定要求
    PIXCESSOR_EVENT_DEFECT_ADDRESS_SET,                                         //!<  画素欠陥補正 アドレス設定要求
    PIXCESSOR_EVENT_DEFECT_AUTO_SET,                                            //!<  画素欠陥補正 自動補正設定要求
    PIXCESSOR_EVENT_DEFECT_LEVEL_SET,                                           //!<  画素欠陥補正 輝度信号レベル設定要求
    PIXCESSOR_EVENT_RGB_MATRIX_SET,                                             //!<  RGBマトリックス 設定要求
#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
    PIXCESSOR_EVENT_NOTIFY_WDR_MODE,                                            //!<  WDRモード変更通知
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
    PIXCESSOR_EVENT_NOTIFY_LINEAR_MODE,                                         //!<  Linear動作モード変更通知
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
    ISP_EVENT_CHROMA_LUT_TABLE_SET,                                             //!<  Chroma LUTテーブル設定要求
    ISP_EVENT_CHROMA_LOW_LUMINANCE_GAIN_SET,                                    //!<  Chroma 低輝度クロマサプレスゲイン設定要求
    ISP_EVENT_CHROMA_EDGE_LUT_TABLE_SET,                                        //!<  Chroma エッジクロマサプレスLUTテーブル設定要求
    ISP_EVENT_SCALER_IMAGE_SIZE_SET,                                            //!<  Scaler スケーラー機能画像サイズ設定要求
    ISP_EVENT_SCALER_NEG_FLIP_SET,                                              //!<  Scaler スケーラー機能ネガポジ・反転設定要求
  };

  // StateMachine用状態テーブル宣言
  static const AXMPixcessor::StateTable state_running[];    //!<  動作中 状態テーブル定義

  // StateMachine用メンバ関数宣言
  /**
   * @brief 擬似フレーム割り込み受信処理を実行する(シミュレータ用)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcFrameIsr(const void *pParam, int size);
#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
  /**
   * @brief AWB用OPD設定を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcAWBOpdSet(const void *pParam, int size);
  /**
   * @brief AE用OPD設定を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcAEOpdSet(const void *pParam, int size);
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
#ifndef ENABLE_OPD_DATA_ISP_TO_AXH
  /**
   * @brief AWB用画像データアドレスを取得する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcAWBImage(const void *pParam, int size);
  /**
   * @brief AE用画像データアドレスを取得する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcAEImage(const void *pParam, int size);
#endif  //  ENABLE_OPD_DATA_ISP_TO_AXH
#ifndef ENABLE_IMAGE_ADDR_ISP_TO_AXH
  /**
   * @brief シェーディング/キズ補正用画像データアドレスを取得する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcCorrectImage(const void *pParam, int size);
#endif  //  ENABLE_IMAGE_ADDR_ISP_TO_AXH
  /**
   * @brief  OBエリア設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBAreaSet(const void *pParam, int size);
  /**
   * @brief　黒レベル設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBDarkLevelSet(const void *pParam, int size);
  /**
   * @brief　OB積算値取得
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBAccumGet(const void *pParam, int size);
  /**
   * @brief　センサー配列パターン選択領域設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBSensorArrayPtnSet(const void* pParam, int size);
  /**
   * @brief　配列パターンへのインデックス値設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBArrayPtnIndexSet(const void* pParam, int size);
  /**
   * @brief　固定黒レベル値設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcOBFixDarkLevelSet(const void* pParam, int size);

  /**
   * @brief ガンマテーブル設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcGammaTblSet(const void *pParam, int size);
  /**
   * @brief　トータルゲイン設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcApertureTotalGainSet(const void *pParam, int size);
  /**
   * @brief 正側・負側リミット設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcAperturePlusMinusLimitSet(const void *pParam, int size);
  /**
   * @brief 高域・中域ゲイン比率設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcApertureOverMidRangeGainRatioSet(const void *pParam, int size);
  /**
   * @brief LUTテーブル設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcApertureLutTblSet(const void *pParam, int size);
  /**
   * @brief コアリング設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcApertureCoringSet(const void *pParam, int size);

  /**
   * @brief レンズシェーディング補正設定
   *
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcLensShadeSet(const void *pParam, int size);

  /**
   * @brief Iridix設定
   *
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcIridixSet(const void *pParam, int size);

  /**
   * @brief クロマサプレス色ゲイン変調LUTテーブル設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcChromaLutTblSet(const void *pParam, int size);

  /**
   * @brief 低輝度クロマサプレスゲイン設定
   *
   * @n
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcChromaLowLuminanceGainSet(const void *pParam, int size);

  /**
   * @brief エッジクロマサプレスLUTテーブル設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcChromaEdgeLutTblSet(const void *pParam, int size);

  /**
   * @brief スケーラー機能画像サイズ設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcScalerImageSizeSet(const void *pParam, int size);
  /**
   * @brief スケーラー機能ネガポジ・反転設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return  実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcScalerNegFlipSet(const void *pParam, int size);

  /**
   * @brief NR一括設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRSet(const void *pParam, int size);

  /**
   * @brief NR Size設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRSizeSet(const void *pParam, int size);

  /**
   * @brief NR Mode設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRModeSet(const void *pParam, int size);

  /**
   * @brief NR MainY設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRMainYSet(const void *pParam, int size);

  /**
   * @brief NR MainC設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRMainCSet(const void *pParam, int size);

  /**
   * @brief NR SubY設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRSubYSet(const void *pParam, int size);

  /**
   * @brief NR SubC設定
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNRSubCSet(const void *pParam, int size);


  /**
   * @brief 画素欠陥補正 一括設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcDefectSet(const void *pParam, int size);

  /**
   * @brief 画素欠陥補正 モード設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcDefectModeSet(const void *pParam, int size);

  /**
   * @brief 画素欠陥補正 モード設定
   *
   * @param   [in]  p_Defect                   画素欠陥補正設定値
   */
  void funcSetDefectMode(ST_DEFECT *p_Defect);

  /**
   * @brief 画素欠陥補正 アドレス設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcDefectAddressSet(const void *pParam, int size);

  /**
   * @brief 画素欠陥補正 アドレス設定
   *
   * @param   [in]  p_Defect                   画素欠陥補正設定値
   */
  void funcSetDefectAddress(ST_DEFECT *p_Defect);

  /**
   * @brief 画素欠陥補正 自動補正設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcDefectAutoSet(const void *pParam, int size);

  /**
   * @brief 画素欠陥補正 自動補正設定
   *
   * @param   [in]  p_Defect                   画素欠陥補正設定値
   */
  void funcSetDefectAuto(ST_DEFECT *p_Defect);

  /**
   * @brief 画素欠陥補正 輝度信号レベル設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcDefectLevelSet(const void *pParam, int size);

  /**
   * @brief 画素欠陥補正 輝度信号レベル設定
   *
   * @param   [in]  p_Defect                   画素欠陥補正設定値
   */
  void funcSetDefectLevel(ST_DEFECT *p_Defect);

  /**
   * @brief RGBマトリックス 設定要求
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcRGBMatrixSet(const void *pParam, int size);

#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
  /**
   * @brief WDRモード変更通知
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNotifyWdrMode(const void *pParam, int size);

  /**
   * @brief Linear動作モード変更通知
   *
   * @param   [in]  pParam                     受信パラメータ
   * @param   [in]  size                       受信パラメータサイズ
   * @return 実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcNotifyLinearMode(const void *pParam, int size);

  /**
   * @brief AWB用の3A画像出力設定を行う
   *
   * @param [in]  frameID   フレームID
   *
   * @n
   */
  void imageif_enc_pic_start_awb(unsigned char frameID);

  /**
   * @brief   AE用の3A画像出力設定を行う
   *
   * @param [in]  frameID   フレーム
   *
   * @n
   */
  void imageif_enc_pic_start_ae(unsigned char frameID);
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
