/**
 * @file   AXCChrono.hpp
 * @brief  AXCChronoクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCCHRONO_HPP_
#define AXCCHRONO_HPP_

#include "AXCStdint.hpp"

/**
 * @class AXCChrono
 * @brief AXC層の Chronoクラス に対し、抽象クラスを提供する
 */
class AXCChrono {
 public:
  /**
   * @brief AXCChronoコンストラクタ
   */
  AXCChrono();

  /**
   * @brief AXCChronoデストラクタ
   */
  virtual ~AXCChrono();

  /**
   * @brief システムから時刻(1985年1月1日0時0分0秒からの経過秒)(※UTC)を取得する
   *
   * @n     エラーが返却された場合、カウンタを初期値(0)のままで戻し、処理結果は戻さない
   * @n     1985年1月1日0時0分0秒丁度が設定されている場合、結果としてカウンタが 0 となるため、
   * @n     エラーが発生したかどうか判断ができない
   * @n
   * @param [out] *sec 秒
   * @param [out] *nsec ナノ秒
   */
  virtual void getTime(ax::INT64_t *sec, ax::INT32_t *nsec) = 0;

  /**
   * @brief 1985年1月1日0時0分0秒からの通算秒(※UTC)をシステムに設定する
   *
   * @param [in] sec 秒
   * @param [in] nsec ナノ秒
   * @return システムへの時刻設定に成功時は 0, エラー時は -1
   */
  virtual ax::INT32_t setTime(ax::INT64_t sec, ax::INT32_t nsec) = 0;

  /** 
   * @brief  年月日時分秒から1985年1月1日0時0分0秒からの通算秒(※UTC)を返却する。
   *
   * @n      現状、int(符号付き32bit) のMax 2038年1月19日3時14分7秒(2147483647) が変換の上限となるが、
   * @n      今後の拡張を考慮し、上記パラメータでのチェックは行っていない
   * @n
   * @param  [in] year   年
   * @param  [in] month  月
   * @param  [in] day    日
   * @param  [in] hour   時
   * @param  [in] minute 分
   * @param  [in] sec    秒
   * @return 1985年1月1日0時0分0秒からの通算秒 (変換エラー時は -1)
   */
  virtual ax::INT64_t convertDateToSec(ax::INT32_t year, ax::INT32_t month,
        ax::INT32_t day, ax::INT32_t hour, ax::INT32_t minute, ax::INT32_t sec) = 0;

};

#endif /* AXCCHRONO_HPP_ */
