/**
 * @file   AXCMutex.hpp
 * @brief  AXCMutexクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCMUTEX_HPP_
#define AXCMUTEX_HPP_

/**
 * @class AXCMutex
 * @brief AXC層の Mutexクラス に対し、抽象クラスを提供する
 *
 */
class AXCMutex {
 public:
  /** @enum  mutexStatus
   *  @brief Mutex実行完了状態
   */
  enum mutexStatus {
    MUTEX_SUCCESS = 0,   ///< 正常終了
    MUTEX_ERROR,         ///< 異常終了
    MUTEX_ERROR_NOINIT,  ///< Mutex未初期化
    MUTEX_ERROR_BUSY,    ///< 既にロック中のため、獲得失敗
  };

  /**
   * @brief AXCMutexコンストラクタ
   */
  AXCMutex();

  /**
   * @brief AXCMutexデストラクタ
   */
  virtual ~AXCMutex();

  /**
   * @brief Mutex初期化
   * @n     プライオリティシーリングや再帰ロック対応が必要になったときは、
   * @n     引数で指定可能となるメソッドを追加する
   * @return mutex初期化状況
   * - MUTEX_SUCCESS 正常終了
   * - MUTEX_ERROR 異常終了
   */
  virtual mutexStatus init() = 0;

  /**
   * @brief Mutexロックの獲得（ブロックあり）
   * @return Mutex獲得状況
   * - MUTEX_SUCCESS 正常終了
   * - MUTEX_ERROR_NOINIT Mutex未初期化
   * - MUTEX_ERROR 異常終了
   */
  virtual mutexStatus lock() = 0;

  /**
   * @brief Mutexロックの獲得（ブロックなし）
   * @return Mutex獲得状況
   * - MUTEX_SUCCESS 正常終了
   * - MUTEX_ERROR_NOINIT Mutex未初期化
   * - MUTEX_ERROR_BUSY 既にロック中のため、獲得失敗
   * - MUTEX_ERROR 異常終了
   */
  virtual mutexStatus trylock() = 0;

  /**
   * @brief Mutexロックの解放
   * @return Mutex解放状況
   * - MUTEX_SUCCESS 正常終了
   * - MUTEX_ERROR_NOINIT Mutex未初期化
   * - MUTEX_ERROR 異常終了
   */
  virtual mutexStatus unlock() = 0;
};

#endif /* AXCMUTEX_HPP_ */
