/**
 * @file   AXCTask.hpp
 * @brief  AXCTaskクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCTASK_HPP_
#define AXCTASK_HPP_

#include <stddef.h>
#include <string>

/**
 * @class AXCTask
 * @brief AXC層の Taskクラス に対し、抽象クラスを提供する
 *
 */
class AXCTask {
 public:
  /** @enum  taskPriority
   *  @brief タスク優先度
   */
  enum taskPriority {
    AXF_PRIORITY_LOW_DEFAULT = 1,
    AXF_PRIORITY_LOW_1 = AXF_PRIORITY_LOW_DEFAULT,
    AXF_PRIORITY_LOW_2,
    AXF_PRIORITY_LOW_3,
    AXF_PRIORITY_LOW_4,
    AXF_PRIORITY_LOW_5,
    AXF_PRIORITY_LOW_6,
    AXF_PRIORITY_LOW_7,
    AXF_PRIORITY_LOW_MAX,
    MODEL_PRIORITY_DEFAULT,
    MODEL_PRIORITY_1 = MODEL_PRIORITY_DEFAULT,
    MODEL_PRIORITY_2,
    MODEL_PRIORITY_3,
    MODEL_PRIORITY_4,
    MODEL_PRIORITY_5,
    MODEL_PRIORITY_6,
    MODEL_PRIORITY_7,
    MODEL_PRIORITY_8,
    MODEL_PRIORITY_9,
    MODEL_PRIORITY_10,
    MODEL_PRIORITY_11,
    MODEL_PRIORITY_12,
    MODEL_PRIORITY_13,
    MODEL_PRIORITY_14,
    MODEL_PRIORITY_15,
    MODEL_PRIORITY_MAX,
    AXF_PRIORITY_HIGH_DEFAULT,
    AXF_PRIORITY_HIGH_1 = AXF_PRIORITY_HIGH_DEFAULT,
    AXF_PRIORITY_HIGH_2,
    AXF_PRIORITY_HIGH_3,
    AXF_PRIORITY_HIGH_4,
    AXF_PRIORITY_HIGH_5,
    AXF_PRIORITY_HIGH_6,
    AXF_PRIORITY_HIGH_7,
    AXF_PRIORITY_HIGH_MAX
  };

  /** @enum  taskStatus
   *  @brief タスク実行完了状態
   */
  enum taskStatus {
    TASK_SUCCESS = 0,     ///< 正常終了
    TASK_ERROR,           ///< 異常終了
    TASK_ERROR_NOCREATE,  ///< タスク未生成
  };

  /**
   * @brief AXCTaskコンストラクタ
   */
  AXCTask();

  /**
   * @brief AXCTaskデストラクタ
   */
  virtual ~AXCTask();

  /**
   * @brief 新しいタスクを生成する。
   * @param [in] name 生成するタスクの名前
   * @param [in] priority 生成するタスクの優先度
   * @param [in] stackSize 生成するタスクのスタックサイズ
   * @param [in] start_routine 生成するタスク関数ポインタ
   * @param [in] arg 生成するタスク関数へ渡す引数
   */
  virtual taskStatus create(const std::string& name, taskPriority priority,
                            size_t stackSize, void *(*start_routine)(void *),
                            void *arg) = 0;

  /**
   * @brief タスクの終了を待つ。
   */
  virtual taskStatus join() = 0;

  /**
   * @brief タスクにキャンセル要求する。
   */
  virtual taskStatus cancel() = 0;

  /**
   * @brief 自タスクを終了する。
   */
  virtual void exitSelf() = 0;
};

#endif /* AXCTASK_HPP_ */
