/**
 * @file   AXCTimerTask.hpp
 * @brief  AXCTimerTaskクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXCTIMERTASK_HPP_
#define AXCTIMERTASK_HPP_

#include "AXCTimerUnit.hpp"

/**
 * @class AXCTimerTask
 * @brief AXC層の TimerTaskクラス に対し、抽象クラスを提供する
 */
class AXCTimerTask {

 public:

  /**
   * @enum
   * @brief TimerTaskメソッド実行完了状態
   */
  enum {
    TIMER_TASK_SUCCESS = 0,
    TIMER_TASK_ERROR_INIT = -10,
    TIMER_TASK_ERROR_WAIT_SELECT = -20,
    TIMER_TASK_ERROR_WAIT_SELECT2 = -21,
    TIMER_TASK_ERROR_WAIT_WAIFLG = -22,
    TIMER_TASK_ERROR_WAIT_PTN = -23,
    TIMER_TASK_ERROR_WAKEUP_SETFLG = -30,
    TIMER_TASK_ERROR_WRITEPIPE_PARAM = -40,
    TIMER_TASK_ERROR_WRITEPIPE_WRITE = -41,
    TIMER_TASK_ERROR_READPIPE_READ = -50,
    TIMER_TASK_ERROR_READPIPE_CMD = -51,
    TIMER_TASK_ERROR_INITCREATE_MUTEXINIT = -60,
    TIMER_TASK_ERROR_INITCREATE_SIMULIST_INIT = -61,
    TIMER_TASK_ERROR_INITCREATE_ALREADY = -62,
    TIMER_TASK_ERROR_INITCREATE_TASKCREATE = -63,
    TIMER_TASK_ERROR_CANCEL_NOTFOUND = -70,
    TIMER_TASK_ERROR_GET_NOTFOUND = -80,
    TIMER_TASK_ERROR_GETLISTTOACTOR_LIST_INIT = -90,
    TIMER_TASK_ERROR_GETLISTTOACTOR_LIST_ADD = -91,
    TIMER_TASK_ERROR_LISTWORK_LOCK = -100,
    TIMER_TASK_ERROR_LISTWORK_UNLOCK = -101,
    TIMER_TASK_ERROR_LISTWORKCANCEL_CANCEL = -1100,
    TIMER_TASK_ERROR_LISTWORKCANCELBYNAME_CANCEL = -120,
    TIMER_TASK_ERROR_LISTWORKGET_GET = -130,
    TIMER_TASK_ERROR_LISTWORKWORKERSUB_SENDEVENT = -140,
  };

  /**
   *  @enum  TimerTaskWakeup
   *  @brief タイマタスク起床要因の種別
   */
  enum TimerTaskWakeup {
    TIMERTASK_WAKEUP_EXPIRE = 0,
    TIMERTASK_WAKEUP_QUIT,
    TIMERTASK_WAKEUP_UPDATE_LIST,
    TIMERTASK_WAKEUP_UPDATE_TIME,
  };

  AXCTimerTask();
  virtual ~AXCTimerTask();

  /**
   * @brief init
   * @n     タイマタスクを初期化する
   * @n
   * @return 設定可否
   */
  virtual int init() = 0;

  /**
   * @brief initWait
   * @n     タイマタスク待機用の初期化処理
   */
  virtual void initWait() = 0;

  /**
   * @brief initWaitTimer
   * @n     タイマタスク待機用の各タイマについての初期化処理
   * @n
   * @param[in]  timer    タイマ
   */
  virtual void initWaitTimer(AXCTimerUnit* timer) = 0;

  /**
   * @brief wait
   * @n     タイマの満了を待機する
   * @n
   * @return 起床要因の種別または負の値のエラーコード
   */
  virtual int wait() = 0;

  /**
   * @brief isExpired
   * @n     指定タイマの満了判定
   * @n
   * @param[in]  timer    タイマ
   * @n
   * @return 指定タイマが満了したか否か
   */
  virtual bool isExpired(AXCTimerUnit* timer) = 0;

  /**
   * @brief wakeUp
   * @n     強制的にタイマタスクを起床させる
   * @n
   * @param[in]  v    起床要因の種別
   * @n
   * @return 設定可否
   */
  virtual int wakeup(TimerTaskWakeup v) = 0;
};

#endif /* AXCTIMERTASK_HPP_ */
