/**
 * @file   AXFMailboxInternal.hpp
 * @brief  AXFMailboxInternalクラスヘッダ
 *
 * @par    Programming Language
 *         C++
 */

#ifndef AXFMAILBOXINTERNAL_HPP_
#define AXFMAILBOXINTERNAL_HPP_

#include "AXFQueueInternal.hpp"

/**
 * @class AXFMailboxInternal
 * @brief スレッド間通信専用のアクタ間通信機能を提供する
 *
 * @n     AXM層は本クラスは直接使用せず、AXFActorクラスの
 * @n     sendメソッド(送信)とStateMachine(受信)を使用する。
 */
class AXFMailboxInternal : public AXFQueueInternal {
  struct queueInfo_t* m_queue;

  // インスタンスのコピーは想定しない(Klockwork指摘対応:CL.FFM.ASSIGN)
  AXFMailboxInternal& operator=(const AXFMailboxInternal&) {
    return *this;
  }

  // インスタンスのコピーは想定しない(Klockwork指摘対応:CL.FFM.COPY)
  AXFMailboxInternal(const AXFMailboxInternal& obj)
      : m_queue(obj.m_queue) {
  }

 public:
  /**
   * @brief AXFMailboxInternalコンストラクタ
   */
  AXFMailboxInternal();

  /**
   * @brief AXFMailboxInternalデストラクタ
   */
  virtual ~AXFMailboxInternal();

  /**
   * @brief Mailbox生成
   * @n
   * @param [in] maxCount 最大データ数
   * @param [in] maxSize 最大データサイズ
   * @return Mailbox生成結果
   */
  virtual queueStatus create(int maxCount, int maxSize);

  /**
   * @brief Mailbox送信処理
   * @n
   * @param [in] eventId イベントID
   * @param [in] data 送信データ
   * @param [in] size 送信データサイズ
   * @return 送信完了データサイズ(送信失敗時は 0未満の値 を返す)
   */
  virtual int send(AXFEvent_t eventId, void* data, int size);

  /**
   * @brief Mailbox受信処理
   * @n
   * @param [out] eventId イベントID
   * @param [out] data 受信データ
   * @param [in] size 受信データサイズ
   * @return 受信完了データサイズ
   */
  virtual int receive(AXFEvent_t* eventId, void* data, int size);

  /**
   * @brief キューに格納されている先頭データのサイズを返す
   * @return データサイズ
   */
  virtual int peek();

  /**
   * @brief エンキューされているデータの個数を返す
   * @return データ数
   */
  virtual int npeek();
};

#endif /* AXFMAILBOXINTERNAL_HPP_ */
