/**
 * @file   AXMIspState.hpp
 * @brief  AXMIspクラス StateMachineオブジェクトヘッダ
 *
 *         Across ISP
 * @par    Programming Language
 *         C++
 */

#ifdef USE_AXBRIDGE_DEMO
#include "../../../../AXF/misc/axbridge-demo/axbridge_demo_param.h"
#endif /* USE_AXBRIDGE_DEMO */

  /**
   * @enum  ispModelState
   * @brief StateMachine用状態定義
   */
  enum ispModelState {
    STATE_0 = 0,                        //!<  待機中
    STATE_FRAME_1ST,                    //!<  1/10フレーム処理
    STATE_FRAME_2ND,                    //!<  2/10フレーム処理
    STATE_FRAME_3RD,                    //!<  3/10フレーム処理
    STATE_FRAME_4TH,                    //!<  4/10フレーム処理
    STATE_FRAME_5TH,                    //!<  5/10フレーム処理
    STATE_FRAME_6TH,                    //!<  6/10フレーム処理
    STATE_FRAME_7TH,                    //!<  7/10フレーム処理
    STATE_FRAME_8TH,                    //!<  8/10フレーム処理
    STATE_FRAME_9TH,                    //!< 9/10フレーム処理
    STATE_FRAME_10TH,                   //!< 10/10フレーム処理
    STATE_100,
    STATE_ADJUST_WB,                    //!<  調整モード : WB調整
    STATE_ADJUST_IRIS,                  //!<  調整モード : IRIS調整
    STATE_ADJUST_SHADE,                 //!<  調整モード : シェーディング補正
    STATE_ADJUST_KIZU,                  //!<  調整モード : キズ補正

    STATE_MAX
  };

  /**
   * @enum  ispModelEvent
   * @brief StateMachine用イベント定義
   */
  enum ispModelEvent {
    //EVENT_START = AXFEVENT_MODEL(0),
    //EVENT_STOP,

    //EVENT_ISR = AXFEVENT_DEVICE_FRAME_ISR,
    EVENT_AE_END = AXFEVENT_MODEL(0),   //!<  AE 周期処理終了イベント
    EVENT_AWB_END,                      //!<  AWB 周期処理終了イベント
    EVENT_SHADE_END,                    //!<  シェーディング補正演算処理修了イベント						// 今回不要　シェーディング補正なし
    EVENT_KIZU_END,                     //!<  キズ補正演算処理修了イベント										// 今回不要　シェーディング補正なし
    EVENT_AE_MODE_CHANGE,               //!<  AE モード変更終了イベント
    EVENT_AWB_MODE_CHANGE,              //!<  AWB モード変更終了イベント
    CORRECTEVENT_SHADE_MODE_CHANGE,     //!<  Correct シェーディング補正 モード変更終了イベント		// 今回不要　シェーディング補正なし
    CORRECTEVENT_WAIT_MODE_CHANGE,      //!<  Correct 待機中 モード変更終了イベント					// 今回不要　シェーディング補正なし
    CORRECTEVENT_KIZU_MODE_CHANGE,      //!<  Correct キズ補正 モード変更終了イベント					// 今回不要　キズ補正なし
    EVENT_AWB_ADJUST_END,               //!<  AWB バラつき調整終了イベント
    EVENT_OB_END,                       //!<  OB 周期処理終了イベント											// 今回不要　OB処理なし
    EVENT_IRQ_FRAMEGEN,                 //!<  フレーム生成完了通知(DSP->ARM)								// 今回不要　ホストCPUへの通知
    EVENT_SPEC_ADDR,                    //!<  設定値保存先通知
    EVENT_NOTIFY_LENS,                  //!<  レンズ情報通知															// 今回不要　レンズ情報不要
#ifdef USE_AXFTEST_COMMAND
    EVENT_PRISCHED_TEST = AXFEVENT_MODEL(1000), //priSched Test Start										// 今回不要　テスト用
    EVENT_PRISCHED_END  = AXFEVENT_MODEL(1001), //priSched Test End											// 今回不要　テスト用
#endif /* USE_AXFTEST_COMMAND */
#ifdef USE_AXBRIDGE_DEMO
    EVENT_AXBRIDGE_TEST_NvrToIsp1 = AXFEVENT_MODEL(AXBRIDGE_DEMO_EVENT_NvrToIsp_1),			// 今回不用　CPU間通信ブリッジ用
    EVENT_AXBRIDGE_TEST_NvrToIsp2 = AXFEVENT_MODEL(AXBRIDGE_DEMO_EVENT_NvrToIsp_2),			// 今回不用　CPU間通信ブリッジ用
#endif /* USE_AXBRIDGE_DEMO */
  };

  /**
   * @enum  IspFrameAe
   * @brief AE 3A画像定義
   */
  enum IspFrameAe {
    ISP_FRAME_AE_3A_0 = 0,              //!<  AE 短時間 3A画像1
    ISP_FRAME_AE_3A_L_0,                //!<  AE 長時間 3A画像1
    ISP_FRAME_AE_3A_1,                  //!<  AE 短時間 3A画像2
    ISP_FRAME_AE_3A_L_1,                //!<  AE 長時間 3A画像2
  };

  /**
   * @enum  IspFrameAwb
   * @brief AWB 3A画像定義
   */
  enum IspFrameAwb {
    ISP_FRAME_AWB_3A = 0,               //!<  AWB 短時間 3A画像
    ISP_FRAME_AWB_3A_L,                 //!<  AWB 長時間 3A画像
  };

  // StateMachine用状態テーブル宣言
  static const AXMIsp::StateTable state_0[];                //!<  待機中 状態テーブル定義
  static const AXMIsp::StateTable state_1[];                //!<  1/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_2[];                //!<  2/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_3[];                //!<  3/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_4[];                //!<  4/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_5[];                //!<  5/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_6[];                //!<  6/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_7[];                //!<  7/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_8[];                //!<  8/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_9[];                //!<  9/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_10[];               //!<  10/10フレーム目処理 状態テーブル定義
  static const AXMIsp::StateTable state_100[];
  static const AXMIsp::StateTable state_1000[];             //!<  調整モード : WB調整 状態テーブル定義
  static const AXMIsp::StateTable state_1100[];             //!<  調整モード : アイリス調整 状態テーブル定義
  static const AXMIsp::StateTable state_1200[];             //!<  調整モード : シェーディング補正 状態テーブル定義
  static const AXMIsp::StateTable state_1300[];             //!<  調整モード : キズ補正 状態テーブル定義

  // StateMachine用メンバ関数宣言
  /**
   * @brief フレーム生成完了通知(DSP->ARM)受信処理
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcFrameGen(const void *pParam, int size);
  /**
   * @brief 設定値保存先アドレス通知を受信し、設定値参照先にそのアドレスを設定する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcSpecAddr(const void *pParam, int size);
  /**
   * @brief 起動直後、1/10フレーム目OPD設定処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcFrameIsr0(const void *pParam, int size);
  /**
   * @brief 1/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0100(const void *pParam, int size);
  /**
   * @brief 1/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0101(const void *pParam, int size);
  /**
   * @brief  1/10フレーム目 AWB演算(色データ計算処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0102(const void *pParam, int size);
  /**
   * @brief  1/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0103(const void *pParam, int size);
  /**
   * @brief 2/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0200(const void *pParam, int size);
  /**
   * @brief 2/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0201(const void *pParam, int size);
  /**
   * @brief 2/10フレーム目 AWB演算(OPDマスク処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0202(const void *pParam, int size);
  /**
   * @brief 2/10フレーム目 OB演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0203(const void *pParam, int size);
  /**
   * @brief 3/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0300(const void *pParam, int size);
  /**
   * @brief 3/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0301(const void *pParam, int size);
  /**
   * @brief 2/10フレーム目 AWB演算(光源枠計算処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0302(const void *pParam, int size);
  /**
   * @brief 3/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0303(const void *pParam, int size);
  /**
   * @brief 4/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0400(const void *pParam, int size);
  /**
   * @brief 4/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0401(const void *pParam, int size);
  /**
   * @brief 4/10フレーム目 AWB演算(重み付け処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0402(const void *pParam, int size);
  /**
   * @brief 4/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0403(const void *pParam, int size);
  /**
   * @brief 5/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0500(const void *pParam, int size);
  /**
   * @brief 5/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0501(const void *pParam, int size);
  /**
   * @brief 5/10フレーム目 AWB(重心計算/調整・収束制御/ゲイン・マトリックス設定)演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0502(const void *pParam, int size);
  /**
   * @brief 5/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0503(const void *pParam, int size);
  /**
   * @brief 6/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0600(const void *pParam, int size);
  /**
   * @brief 6/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0601(const void *pParam, int size);
  /**
   * @brief 6/10フレーム目 AWB演算(色データ計算)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0602(const void *pParam, int size);
  /**
   * @brief 6/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0603(const void *pParam, int size);
  /**
   * @brief 7/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0700(const void *pParam, int size);
  /**
   * @brief 7/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0701(const void *pParam, int size);
  /**
   * @brief 7/10フレーム目 AWB演算(OPDマスク処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0702(const void *pParam, int size);
  /**
   * @brief 7/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0703(const void *pParam, int size);
  /**
   * @brief 8/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0800(const void *pParam, int size);
  /**
   * @brief 8/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0801(const void *pParam, int size);
  /**
   * @brief 8/10フレーム目 AWB演算(光源枠判定処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0802(const void *pParam, int size);
  /**
   * @brief 8/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0803(const void *pParam, int size);
  /**
   * @brief 9/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0900(const void *pParam, int size);
  /**
   * @brief 9/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0901(const void *pParam, int size);
  /**
   * @brief 9/10フレーム目 AW演算(重み付け処理)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0902(const void *pParam, int size);
  /**
   * @brief 9/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0903(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 割り込み受信処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A00(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 AE周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A01(const void *pParam, int size);
  /**
   * @brief 9/10フレーム目 AWB演算(重心計算/調整・収束制御/ゲイン・マトリックス設定)終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A02(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 OB周期演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A03(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 AEモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A08(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 AWBモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A09(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 シェーディング補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A0A(const void *pParam, int size);
  /**
   * @brief 10/10フレーム目 キズ補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0A0B(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：ばらつき調整の要求処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1000(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：バラつき調整終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1001(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：AEモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1008(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：AWBモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1009(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：シェーディング補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix100A(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)：キズ補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix100B(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：IRIS調整の要求処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1100(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：AE アイリス調整終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1101(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：AEモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1108(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：AWBモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1109(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：シェーディング補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix110A(const void *pParam, int size);
  /**
   * @brief 調整モード(IRIS調整)：キズ補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix110B(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：シェーディング補正の要求処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1200(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：AEシャッター制御終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1201(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：シェーディング補正演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1202(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：AEモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1208(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：AWBモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1209(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：Correct 待機中モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix120A(const void *pParam, int size);
  /**
   * @brief 調整モード(シェーディング補正)：correct キズ補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix120B(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：キズ補正の処理要求を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1300(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：キズ補正演算終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1301(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：AEアイリス調整終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1302(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：AEモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1308(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：AWBモード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1309(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：Correct シェーディング補正モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix130A(const void *pParam, int size);
  /**
   * @brief 調整モード(キズ補正)：Correct 待機中モード変更終了処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return	実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix130B(const void *pParam, int size);

  ax::actorFuncStatus funcFrameIsr100(const void *pParam, int size);
#ifdef USE_AXFTEST_COMMAND
  ax::actorFuncStatus funcPriSchedTest(const void *pParam, int size);
  ax::actorFuncStatus funcPriSchedEnd(const void *pParam, int size);
#endif /* USE_AXFTEST_COMMAND */
#ifdef USE_AXBRIDGE_DEMO
  ax::actorFuncStatus funcNvrToIsp1(const void *pParam, int size);
  ax::actorFuncStatus funcNvrToIsp2(const void *pParam, int size);
  void sendNvrTest();
#endif /* USE_AXBRIDGE_DEMO */
#ifndef ENABLE_OPD_CONFIG_ISP_TO_AXH
  void setImgWDRMode(bool isWdrMode);
#endif  //  ENABLE_OPD_CONFIG_ISP_TO_AXH
  void getEvalDataForAf(void);
  ax::actorFuncStatus recvLenzInfo(const void *pParam, int size);               // レンズ情報を受信
void funcTest(void);	// for debug

  int   state;                          ///!  次期ステータス格納用

  /**
   * @brief AWBのOPD設定を行う
   *
   * @param [in] mode AWBモード
   */
  void  setOpdConfigAwb(IspFrameAwb mode);
  /**
   * @brief AEのOPD設定を行う
   *
   * @param [in] mode AEモード
   */
  void  setOpdConfigAe(IspFrameAe mode);

  /**
   * @brief　OB変更チェック
   */
  void  chkOBChange();                                      // OB変更チェック
  /**
   * @brief　アパーチャ補正変更チェック
   */
  void	 chkApertureChange();                                 // アパーチャ補正変更チェック
  /**
   * @brief　アパーチャ補正変更チェック
   */
  void	 chkApertureChangeEx();                               // アパーチャ拡張補正変更チェック

  /**
   * @brief クロマサプレス変更チェック
   */
  void	 chkChromaChange();                                  // クロマサプレス変更チェック

  /**
   * @brief スケーラー制御変更チェック
   */
  void	 chkScalerChange();                                  // スケーラー制御変更チェック

  /**
   * @brief ImageIF 動作モード設定
   */
  void  setImageIFOpeMode();                                // ImageIF 動作モード設定

  /**
   * @brief バンク切り替え用カウンタ値を、第1フレームを指す値にする
  */
  void  setBankCounterAtFirstFrame();

  /**
   * @brief Pixcessor メモリバンク設定
   */
  void  setPixcessorMemBank();                              // ImageIF メモリバンク設定

  /**
   * @brief AWB用画像データ要求
   *
   * @param [in] msg 送信用メッセージ
   */
  void  getOpdDataForAwb(ST_MSG msg);
  /**
   * @brief AE用画像データ要求
   *
   * @param [in] msg 送信用メッセージ
   */
  void  getOpdDataForAe(ST_MSG msg);
  /**
   * @brief シェーディング補正用画像データ要求
   *
   * @n
   */
  void  getImageDataForShading();
  /**
   * @brief キズ補正用画像データ要求
   *
   * @n
   */
  void  getImageDataForKizu();
  /**
   * @brief　OB処理実行
   */
  void  execObProc();

  /**
   * @brief ISPモードのモード変更処理を行う
   */
  void matrix0Axx_checkModeStatus();

  /**
   * @brief AWBモード時のモード変更処理を行う
   *
   * @return ステータス変更の有無
   */
  bool  checkAWBMode_IspModeRun();

#ifdef USE_AXFTEST_COMMAND
  int m_prischecSuccess;
  int m_prischecCount;
#endif /* USE_AXFTEST_COMMAND */
#ifdef PARASOFT_CPPTEST
  int CPPTest_WORK;
  int CPPTest_Param[10];
  ST_SCALER CPPTest_execSclCtrl;
#endif

  bool isWdrMode;                                           //!<  WDRモードフラグ
  bool isLinearMode;                                        //

  unsigned short Luminance;                                 //  明るさ情報(AE -> ISP -> AWB)

