/**
 * @file       	AXMIspAeValue.hpp
 * @brief      	AE演算用クラス群ヘッダ
 *
 * @par			言語
 *              C++
 */
#ifndef AXMISPAEVALUE_H_
#define AXMISPAEVALUE_H_

#include "AXMISPtype.hpp"

/**
 * @class  AXMIspAePercent
 * @brief     パーセント計算クラス
 *
 * @n			
 */
class AXMIspAePercent {
 protected:
  struct {
    USHORT b :7;
  } AePercent;

 public:
  /**
   * @brief  オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAePercent();
  /**
   * @brief  オブジェクトを開放する
   *
   * @n
   */
  virtual ~AXMIspAePercent();
  /**
   * @brief    パーセント計算クラス ％値設定
   *
   * @n     小数点以下第２位までの値を100倍して設定する
   * @param [in]  percent	パーセント
   */
  void set(USHORT percent);
  /**
   * @brief     パーセント計算クラス 代入演算子
   *
   * @n     小数点以下第２位までの値を100倍した値を代入する
   * @param[in] obj	右辺
   * @return    オブジェクト参照
   */
  AXMIspAePercent& operator=(const short& obj);
  /**
   * @brief     ％値表示
   *
   * @n     小数点以下第２位までの少数をX.XXの書式で表示する
   */
  void print() const;
  /**
   * @brief    パーセント計算クラス ％値取得
   *
   * @n     小数点以下第２位までの値を100倍された値を取得する
   * @return ％値
   */
  USHORT get(void) const;
};

/**
 * @class  AXMIspAeValue
 * @brief     AE演算用少数クラス
 *
 * @n			
 */
class AXMIspAeValue {
 public:
  /**
   * @brief  オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAeValue();
  /**
   * @brief  オブジェクトを開放する
   *
   * @n
   */
  virtual ~AXMIspAeValue();

  /**
   * @brief    AE演算用少数クラス+演算子オーバーロード
   *
   * @param    [in] obj	加数
   * @return   加算結果
   */
  AXMIspAeValue operator+(const AXMIspAeValue& obj);		// +演算子オーバーロード
  /**
   * @brief    AE演算用少数クラス+=演算子オーバーロード
   *
   * @param    [in] obj	加数
   * @return   加算結果
   */
  AXMIspAeValue& operator+=(const AXMIspAeValue& obj);	// +=演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス-演算子オーバーロード
   *
   * @param    [in] obj	減数
   * @return   減算結果
   */
  AXMIspAeValue operator-(const AXMIspAeValue& obj);		// -演算子オーバーロード
  /**
   * @brief    AE演算用少数クラス-=演算子オーバーロード
   *
   * @param   [in] obj	減数
   * @return  減算結果
   */
  AXMIspAeValue& operator-=(const AXMIspAeValue& obj);	// -=演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス*パーセント演算子オーバーロード
   *
   * @param     [in] obj	パーセント
   * @return    パーセント結果
   */
  AXMIspAeValue operator*(const AXMIspAePercent& obj);  // *演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス*演算子オーバーロード
   *
   * @param     [in] obj	AE演算用少数
   * @return    乗算結果
   */
  AXMIspAeValue operator*(const AXMIspAeValue& obj);		// *演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス/演算子オーバーロード
   *
   * @param     [in] obj	除数
   * @return    除算結果
   */
  AXMIspAeValue operator/(const USHORT& obj);				// /演算子オーバーロード

  /**
   * @brief     AE演算用少数クラス>演算子オーバーロード
   *
   * @param     [in] obj	右辺
   * @retval    true  左辺＞右辺
   * @retval    false 左辺≦右辺
   */
  bool operator>(const AXMIspAeValue& obj);				// >演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス>=演算子オーバーロード
   *
   * @param     [in] obj	右辺
   * @retval    true  左辺≧右辺
   * @retval    false 左辺＜右辺
   */
  bool operator>=(const AXMIspAeValue& obj);				// >=演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス<演算子オーバーロード
   *
   * @param     [in] obj	右辺
   * @retval    true  左辺＜右辺
   * @retval    false 左辺≧右辺
   */
  bool operator<(const AXMIspAeValue& obj);				// <演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス<=演算子オーバーロード
   *
   * @param     [in] obj	右辺
   * @retval    true  左辺≦右辺
   * @retval    false 左辺＞右辺
   */
  bool operator<=(const AXMIspAeValue& obj);				// <=演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス==演算子オーバーロード
   *
   * @param     [in] obj	右辺
   * @retval    true  左辺==右辺
   * @retval    false 左辺!=右辺
   */
  bool operator==(const AXMIspAeValue& obj);				// ==演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス set関数
   *
   * @param     [in] intval 整数部
   * @param     [in] fracval 少数部
   * @n 負数の設定不可
   */
  void set(USHORT intval, USHORT fracval);				// set関数(整数部と小数部を指定）
  /**
   * @brief     AE演算用少数クラス set関数
   *
   * @param     [in] longval (整数部*10)+ USHORT 少数部
   * @n 負数の設定可
   */
  void set(long longval);									// set関数(10倍したlongデータを指定)
  /**
   * @brief     AE演算用少数クラス 代入演算子
   *
   * @param     [in] obj 右辺
   * @n 負数の設定可
   */
  AXMIspAeValue& operator=(const long& obj);				// =演算子オーバーロード
  /**
   * @brief     AE演算用少数クラス get関数
   *
   * @return    LONG (整数部*10)+ USHORT 少数部
   */
  short get(void) const;									// 整数部8bit　小数部：4bit　取得
  /**
   * @brief     AE演算用少数クラス long変換関数
   *
   * @return    LONG (整数部*10)+ USHORT 少数部
   */
  long conv2Long(void) const;								// longデータに変換
  /**
   * @brief     AE演算用少数クラス 整数部取得
   *
   * @return    USHORT 整数部
   */
  USHORT getIntval(void) const;
  /**
   * @brief     AE演算用少数クラス 少数取得
   *
   * @return    USHORT 少数部
   */
  USHORT getFracval(void) const;
  /**
   * @brief     AE演算用少数クラス 符号取得
   *
   * @return    USHORT 符号
   * @n 引き算で１に設定される場合がある
   */
  USHORT getSign(void) const;
  /**
   * @brief     AE演算用少数クラス 表示
   *
   * @n 小数点以下第1位までの少数をXXX.Xの書式で表示する
   */
  void print() const;
 private:
  const static long AE_MAX_VALUE = 2559;
  struct {	// bit field
    USHORT sign :1;
    USHORT FILLER :3;
    USHORT intval :8;
    USHORT fracval :4;
  } AeVal;

};

#endif /* AXMISPAEVALUE_H_ */
