/**
 * @file	AXMIspAwbSettings.hpp
 * @brief	AXMIspAwb設定データクラス群ヘッダ
 *
 * @par		言語
 *			C++
 */

#ifndef AXMISPAWBSETTINGS_HPP_
#define AXMISPAWBSETTINGS_HPP_


/**
 * @class	ImgPoint
 * @brief	画像格納用クラス
 */
class ImgPoint {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  ImgPoint();

  /**
   * @brief 水平,垂直座標を設定する
   *
   * @param [in]  hData 水平座標
   * @param [in]  vData 垂直座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(SHORT hData, SHORT vData);

  SHORT h;							//	水平座標
  SHORT v;							//	垂直座標

};

/**
 * @class	ColPoint
 * @brief	色画像座標格納用クラス
 */
class ColPoint {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  ColPoint();

  /**
   * @brief 色座標を設定する
   *
   * @param [in]  bgData  B/G軸座標データ
   * @param [in]  rgData  R/G軸座標データ
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(SHORT bgData, SHORT rgData);

  SHORT bg;							//	B/G軸座標
  SHORT rg;							//	R/G軸座標

};

/**
 * @class	AreaColPoint
 * @brief	光源座標格納用クラス
 */
class AreaColorPoint {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AreaColorPoint();

  /**
   * @brief 光源座標を設定する
   *
   * @param [in]  index 光源枠座標番号
   * @param [in]  bgData  B/G軸座標
   * @param [in]  rgData  R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(UCHAR index, SHORT bgData, SHORT rgData);

  AWBLightSource ID;										//	光源枠番号
  USHORT Weight;									//	光源枠重み
  //	光源座標
  //		0. 左上
  //		1. 右上
  //		2. 右下
  //		3. 左下
  ColPoint CPnt[AWB_LightSourcePnt];				//	光源枠座標

};

/**
 * @class	LightSourceWeight
 * @brief	光源枠格納用クラス
 */
class LightSourceWeight {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  LightSourceWeight();

  UCHAR WeightL;					// 低輝度判定重み
  UCHAR WeightH;					// 高輝度判定重み
  USHORT LimitL;						// 低輝度判定上限
  USHORT LimitH;						// 高輝度判定下限

};

/**
 * @class	WBGain
 * @brief	ホワイトバランス格納用クラス
 */
class WBGain {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  WBGain();

  USHORT RGain;						// Rホワイトバランス
  USHORT BGain;						// Bホワイトバランス

};

/**
 * @class	AdjustMatrix
 * @brief	カラーマトリックス調整係数格納用クラス
 */
class AdjustMatrix {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AdjustMatrix();
  //	Destructor
  //virtual ~AdjustMatrix(){};

  USHORT AJCF11;						// Cbクロマゲイン調整係数
  USHORT AJCF12;						// Cbマトリックスゲイン調整係数
  USHORT AJCF21;						// Crマトリックスゲイン調整係数
  USHORT AJCF22;						// Crクロマゲイン調整係数

};

/**
 * @class	ColorMatrix
 * @brief	カラーマトリックス係数格納用クラス
 */
class ColorMatrix {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  ColorMatrix();

  USHORT CC22;						// Cbクロマゲイン
  USHORT CC23;						// Cbマトリックスゲイン
  USHORT CC33;						// Crクロマゲイン
  USHORT CC32;						// Crマトリックスゲイン

};

/**
 * @class	AXMIspAwbSettingsOnePush
 * @brief	［AWB-5］全引込み実行設定値格納用クラス
 */
class AXMIspAwbSettingsOnePush {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsOnePush();
  //	Destructor
  //virtual	~AXMIspAwbSettingsOnePush(){};

  /**
   * @brief 全引き込み実行レジスタに設定する。設定後は、レジスタの内容をクリアする。
   *
   * @param [in]  data  全引込み実行レジスタ設定内容
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setStart(UCHAR& data);

  /**
   * @brief 全引込み実行時OPDエリアを設定する
   *
   * @param [in]  index 0:開始座標, 1:終了座標
   * @param [in]  data  画像座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setOPDArea(AWBOnePushPnt index, ImgPoint data);

  /**
   * @brief 全引込み時光源引き込み範囲を設定する
   *
   * @param [in]  data  引込み範囲
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setRange(AWBOnePushRange data);

  /**
   * @brief 引込み速度を設定する
   *
   * @param [in]  data  引込み速度
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setSpeed(UCHAR data);

//private:
  UCHAR Start;                                              //  全引込み実行
  ImgPoint OPDArea[AWBOnePushPnt_Max];                      //  全引込み実行時OPDエリア
  AWBOnePushRange Range;                                    //  全引込み時光源引込み範囲
  UCHAR Speed;                                              //  全引込み時引込み速度

};

/**
 * @class	AXMIspAwbSettingsLightRange
 * @brief	［AWB-6］AWB光源枠引込み範囲設定値格納用クラス
 */
class AXMIspAwbSettingsLightRange {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsLightRange();
  //	Destructor
  //virtual	~AXMIspAwbSettingsLightRange(){};

  /**
   * @brief 低色温度側引込み範囲を設定する
   *
   * @param [in]  data  低色温度側引込み範囲
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLTempAdjust(USHORT data);

  /**
   * @brief 高色温度側引込み範囲を設定する
   *
   * @param [in]  data  高色温度側引込み範囲
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setHTempAdjust(USHORT data);

  /**
   * @brief Mg方向引込み範囲を設定する
   *
   * @param [in]  data  Mg方向引込み範囲
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setMgDirAdjust(USHORT data);

  /**
   * @brief G方向引込み範囲を設定する
   *
   * @param [in]  data  G方向引込み範囲
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setGDirAdjust(USHORT data);

  USHORT LTempAdjust;                   //  低色温度側引込み範囲
  USHORT HTempAdjust;                   //  高色温度側引込み範囲
  ColPoint CPnt[AWB_LightRange_Num];    //  引込みポイント座標
  USHORT MgDirAdjust;                   //  Mg方向引込み範囲
  USHORT GDirAdjust;                    //  G方向引込み範囲

};

/**
 * @class	AXMIspAwbSettingsOffset
 * @brief	［AWB-7］WBオフセット調整設定値格納用クラス
 */
class AXMIspAwbSettingsOffset {

 public:
  /**
   * @brief オフセット調整設定値を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  bg  B/G軸座標
   * @param [in]  rg  R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setOffset(AWBLightSource index, SHORT bg, SHORT rg);

  ColPoint Pnt[AWBLightSource_Max];

};

/**
 * @class	AXMIspAwbSettingsAdjustMatrix
 * @brief	［AWB-8］各光源枠カラーマトリックス調整設定値格納用クラス
 */
class AXMIspAwbSettingsAdjustMatrix {

 public:
  /**
   * @brief Cbクロマゲイン調整係数を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  AJCF11  Cbクロマゲイン調整係数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setAJCF11(AWBLightSource index, USHORT AJCF11);

  /**
   * @brief Cbマトリックスゲイン調整係数を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  AJCF12  Cbマトリックスゲイン調整係数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setAJCF12(AWBLightSource index, USHORT AJCF12);

  /**
   * @brief Crマトリックスゲイン調整係数を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  AJCF21  Crマトリックスゲイン調整係数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setAJCF21(AWBLightSource index, USHORT AJCF21);

  /**
   * @brief Crクロマゲイン調整係数を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  AJCF22  Crクロマゲイン調整係数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setAJCF22(AWBLightSource index, USHORT AJCF22);

  AdjustMatrix Matrix[AWBLightSource_Max];

};

/**
 * @class	AXMIspAwbSettingsWB
 * @brief	［AWB-9］マニュアルモード時ゲイン指定設定値格納用クラス
 */
class AXMIspAwbSettingsWB : public WBGain {

 public:
  /**
   * @brief マニュアルモード時に使用するRゲインを設定する
   *
   * @param [in]  RGainData Rゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setRGain(USHORT RGainData);

  /**
   * @brief マニュアルモード時に使用するBゲインを設定する
   *
   * @param [in]  BGainData Bゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setBGain(USHORT BGainData);

};

/**
 * @class	AXMIspAwbSettingsConvergence
 * @brief	［AWB-10］AWB遷移特性設定クラス
 */
class AXMIspAwbSettingsConvergence {

 public:
  /**
   * @brief オブジェクトを初期化する
   */
  AXMIspAwbSettingsConvergence();

  /**
   * @brief 遷移速度を設定する
   *
   * @param [in]  SpeedData 遷移速度:フレーム数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setSpeed(USHORT SpeedData);

  /**
   * @brief 引込み速度を設定する
   *
   * @param [in]  LightTHData 照明変化検知閾値:lx
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLightTHData(USHORT LightTHData);

  /**
   * @brief 安定性確認閾値を設定する
   *
   * @param [in]  StableTHData  安定性確認閾値:lx
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setStableTHData(USHORT StableTHData);

  /**
   * @brief 安定性確認期間を設定する
   *
   * @param [in]  StableTermData  安定性確認期間:フレーム数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setStableTerm(USHORT StableTermData);

  USHORT Speed;						//	引込み速度       : フレーム数
  USHORT LightTH;					//	照明変化検知閾値 : lx
  USHORT StableTH;					//	安定性確認閾値   : lx
  USHORT StableTerm;					//	安定性確認期間   : フレーム数

};

/**
 * @class	AXMIspAwbSettingsLightSourceWeight
 * @brief	［AWB-11］光源別光源枠重み付け設定クラス
 */
class AXMIspAwbSettingsLightSourceWeight {

 public:
  /**
   * @brief 低輝度時重みを設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  WeightLData 低輝度時重み
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setWeightL(AWBLightSource index, UCHAR WeightLData);

  /**
   * @brief 高輝度時重みを設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  WeightHData 高輝度時重み
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setWeightH(AWBLightSource index, UCHAR WeightHData);

  /**
   * @brief 低輝度時判定上限を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  LimitLData  低輝度時判定上限
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLimitL(AWBLightSource index, USHORT LimitLData);		// 低輝度時判定上限 設定


  /**
   * @brief 高輝度時判定上限を設定する
   *
   * @param [in]  index 光源番号
   * @param [in]  LimitHData  高輝度時判定下限
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLimitH(AWBLightSource index, USHORT LimitHData);		// 高輝度時判定下限 設定

  LightSourceWeight Weight[AWBLightSource_Max];

};

/**
 * @class	AXMIspAwbSettingsOPDSet
 * @brief	［AWB-13］OPD設定値格納用クラス
 */
class AXMIspAwbSettingsOPDSet {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsOPDSet();

  /**
   * @brief 水平分割数を設定する
   *
   * @param [in]  HDivData  水平分割数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setHDiv(USHORT HDivData);

  /**
   * @brief 垂直分割数を設定する
   *
   * @param [in]  VDivData  垂直分割数
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setVDiv(USHORT VDivData);

  /**
   * @brief 積算対象信号の上限を設定する
   *
   * @param [in]  HLimitData  積算対象信号上限
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setHLimit(USHORT HLimitData);

  /**
   * @brief 積算対象信号の下限を設定する
   *
   * @param [in]  LLimitData  積算対象信号下限
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLLimit(USHORT LLimitData);

  USHORT HDiv;						//	水平分割数
  USHORT VDiv;						//	垂直分割数
  USHORT HLimit;						//	積算対象信号上限
  USHORT LLimit;						//	積算対象信号下限

};

/**
 * @class	AXMIspAwbSettingsOPDWeight
 * @brief	［AWB-14］OPD枠重み付け設定値格納用クラス
 */
class AXMIspAwbSettingsOPDWeight {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsOPDWeight();

  /**
   * @brief エリア座標V1(垂直方向の座標のみ)を設定する
   *
   * @param [in]  V1Data  エリア座標V1（垂直方向座標のみ）
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setV1(USHORT V1Data);

  /**
   * @brief エリア座標H2(水平方向の座標のみ)を設定する
   *
   * @param [in]  H2Data  エリア座標H2(水平方向座標のみ)
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setH2(USHORT H2Data);

  /**
   * @brief エリア座標H3(水平方向の座標のみ)を設定する
   *
   * @param [in]  H3Data  エリア座標H3(水平方向座標のみ)
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setH3(USHORT H3Data);

  /**
   * @brief エリア座標V4(垂直方向の座標のみ)を設定する
   *
   * @param [in]  V4Data  エリア座標V4(垂直方向座標のみ)
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setV4(USHORT V4Data);

  /**
   * @brief OPDエリアごとの重みを設定する
   *
   * @param [in]  index OPDエリア番号
   * @param [in]  WeightModeData  重み番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setWeightMode(AWB_OPDArea index, AWBWeightMode WeightModeData);

  USHORT V1;												//	エリア座標v1
  USHORT H2;												//	エリア座標h2
  USHORT H3;												//	エリア座標h3
  USHORT V4;												//	エリア座標v4
  AWBWeightMode WeightMode[AWB_OPDArea_Idx_Max];	//	重み

};

/**
 * @class	AXMIspAwbSettingsOPDMask
 * @brief	［AWB-15］OPD枠AWBマスク領域設定クラス
 */
class AXMIspAwbSettingsOPDMask {

 public:
  /**
   * @brief マスク座標を設定する
   *
   * @param [in]  index 座標index番号
   * @param [in]  h 水平方向座標
   * @param [in]  v 垂直方向座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(UCHAR index, SHORT h, SHORT v);

  ImgPoint Pnt[AWB_OPDMaskSetMax];

};

/**
 * @class	AXMIspAwbSettingsSpectrumAdjust
 * @brief	［AWB-16］センサー分光バラツキ調整クラス
 */
class AXMIspAwbSettingsSpectrumAdjust {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsSpectrumAdjust();

  /**
   * @brief 低温側調整ポイントを設定する
   *
   * @param [in]  bg  B/G軸座標
   * @param [in]  rg  R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLowTempPoint(SHORT bg, SHORT rg);

  /**
   * @brief 高温側調整ポイントを設定する
   *
   * @param [in]  bg  B/G軸座標
   * @param [in]  rg  R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setHighTempPoint(SHORT bg, SHORT rg);

  UCHAR LColorTempStart;		//	低色温度側調整起動レジスタ
  UCHAR HColorTempStart;		//	高色温度側調整起動レジスタ
  ColPoint LowTempPnt;				//	低温側調整ポイント
  ColPoint HighTempPnt;			//	高温側調整ポイント

};

/**
 * @class	AXMIspAwbSettingsLightSource
 * @brief	［AWB-17］光源枠詳細調整設定クラス
 */
class AXMIspAwbSettingsLightSource {

 public:
  /**
   * @brief 光源毎に光源枠座標(4点)を設定する
   *
   * @param [in]  index1  光源番号
   * @param [in]  index2  光源枠座標番号
   * @param [in]  bg  B/G軸座標
   * @param [in]  rg  R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(AWBLightSource index1, UCHAR index2, SHORT bg, SHORT rg);

  AreaColorPoint AreaOriginal[AWBLightSource_Rolling_Max];  //  光源枠座標設定値
  AreaColorPoint Area[AWBLightSource_Rolling_Max];          //  光源枠座標補正値(演算用)
  //	 0.白熱灯枠座標
  //	 1.電球色蛍光灯座標
  //	 2.白色蛍光灯枠座標
  //	 3.昼白色蛍光灯枠座標
  //	 4.昼光色蛍光灯枠座標
  //	 5.昼光枠座標
  //	 6.曇天/日陰枠座標
  //	 7.水銀灯枠座標
  //	 8.高圧ナトリウムランプ枠座標
  //	 9.電球色LED枠座標
  //	10.白色LED枠座標
  //	11.昼光色LED枠座標
  //	12.白色蛍光灯枠座標(カラーローリング検知時)
  //	13.昼白色蛍光灯枠座標(カラーローリング検知時)
  //	14.昼光色蛍光灯枠座標(カラーローリング検知時)

};

/**
 * @class	AXMIspAwbSettingsLightDefault
 * @brief	［AWB-18］光源デフォルト位置設定クラス
 */
class AXMIspAwbSettingsLightDefault {

 public:
  /**
   * @brief 光源毎にデフォルト座標を設定する
   *
   * @param [in]  index デフォルト座標番号
   * @param [in]  bg	B/G軸座標
   * @param [in]  rg	R/G軸座標
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setData(AWBLightDefault index, SHORT bg, SHORT rg);

  ColPoint CPnt[AWBLightDefault_Max];					//	光源デフォルト座標
  //	 0.電球色蛍光灯デフォルト座標
  //	 1.白色蛍光灯デフォルト座標
  //	 2.昼白色蛍光灯デフォルト座標
  //	 3.昼光色蛍光灯デフォルト座標
  //	 4.水銀灯デフォルト座標
  //	 5.高圧ナトリウムランプデフォルト座標
  //	 6.電球色LEDデフォルト座標
  //	 7.白色LEDデフォルト座標
  //	 8.昼光色LEDデフォルト座標
  //	 9. 1800Kポイント座標(設定により取得)
  //	10. 3200Kポイント座標(調整により取得)
  //	11. 4200Kポイント座標(設定により取得)
  //	12. 5200Kポイント座標(設定により取得)
  //	13. 5800Kポイント座標(調整により取得)
  //	14. 7000Kポイント座標(設定により取得)
  //	15.18000Kポイント座標(設定により取得)

};

/**
 * @class	AXMIspAwbSettingsOPDTrust
 * @brief	［AWB-19］OPD信頼性判定レベル設定値格納用クラス
 */
class AXMIspAwbSettingsOPDTrust {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsOPDTrust();

  /**
   * @brief 信頼性閾値を設定する
   *
   * @param [in]  THData  信頼性閾値
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setTH(USHORT THData);

  /**
   * @brief 低信頼性時の処理を選択する
   *
   * @param [in]  SelData 低信頼性時処理番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setSel(AWBLowTrust SelData);

  /**
   * @brief OPDデータゼロ時の処理を選択する
   *
   * @param [in]  ZeroData  光源番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setZero(AWBLightSource ZeroData);

  USHORT TH;												//	信頼性閾値
  AWBLowTrust Sel;										//	低信頼性時の処理選択
  //		0.ホールド
  //		1.近い光源デフォルト
  AWBLightSource Zero;									//	OPDデータゼロ時処理選択
  //		 0.白熱灯デフォルト座標
  //		 1.電球色蛍光灯デフォルト座標
  //		 2.白色蛍光灯デフォルト座標
  //		 3.昼白色蛍光灯デフォルト座標
  //		 4.昼光色蛍光灯デフォルト座標
  //		 5.昼光デフォルト座標
  //		 6.水銀灯デフォルト座標
  //		 7.高圧ナトリウムランプデフォルト座標
  //		 8.電球色LEDデフォルト座標
  //		 9.白色LEDデフォルト座標
  //		10.昼光LEDデフォルト座標
  //		11.黒体放射 7000K座標

};

/**
 * @class	AXMIspAwbSettingsColorMatrix
 * @brief	［Img-5.2.2］ カラーマトリックス設定値格納用クラス
 */
class AXMIspAwbSettingsColorMatrix {

 public:
  /**
   * @brief オブジェクトを初期化する
   *
   * @n
   */
  AXMIspAwbSettingsColorMatrix();

  /**
   * @brief 独立モードのON/OFFを設定する
   *
   * @param [in]  ModeData  モード状態(ON/OFF)
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setMode(UCHAR ModeData);

  /**
   * @brief Cbクロマゲインを設定する
   *
   * @param [in]  index 象限番号
   * @param [in]  CC22Data  Cbクロマゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCC22(UCHAR index, USHORT CC22Data);

  /**
   * @brief Cbマトリックスゲインを設定する
   *
   * @param [in]  index 象限番号
   * @param [in]  CC23Data  Cbマトリックスゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCC23(UCHAR index, USHORT CC23Data);

  /**
   * @brief Crクロマゲインを設定する
   *
   * @param [in]  index 象限番号
   * @param [in]  CC33Data  Crクロマゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCC33(UCHAR index, USHORT CC33Data);

  /**
   * @brief Crマトリックスゲインを設定する
   *
   * @param [in]  index 象限番号
   * @param [in]  CC32Data  Crマトリックスゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCC32(UCHAR index, USHORT CC32Data);

  /**
   * @brief Yゲインを設定する
   *
   * @param [in]  CC11Data  Yゲイン
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCC11(USHORT CC11Data);

  /**
   * @brief Yオフセットを設定する
   *
   * @param [in]  COF1Data  Yオフセット
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setCOF1(USHORT COF1Data);

  UCHAR Mode;                                               //  独立モードON/OFF(0. OFF, 1. ON)
  ColorMatrix Matrix[AWB_ColorMatrixMax];                   //  クロマ/マトリックスゲイン
  USHORT CC11;                                              //  Yゲイン
  USHORT COF1;                                              //  Yオフセット

};

/**
 * @class	AXMIspAwbSettings
 * @brief	AWB関連設定値格納用クラス
 */
class AXMIspAwbSettings {

 public:
  /**
   * @brief オブジェクトを初期化する
   */
  AXMIspAwbSettings();

  /**
   * @brief AWBモードを設定する
   *
   * @param [in]  data  AWBモード番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setAWBMode(AWBMode data);

  /**
   * @brief ATWシーンモード番号を設定する
   *
   * @param [in]  data  ATWシーンモード番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setATWSceaneMode(ATWSceaneMode data);

  /**
   * @brief マニュアルモード時の色温度を指定する
   *
   * @param [in]  data  色温度
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setColorTemp(USHORT data);

  /**
   * @brief マニュアルモード時の照明を指定する
   *
   * @param [in]  data  光源番号
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setLightSelect(AWBLightSelect data);

  /**
   * @brief ATWモードの停止閾値を設定する
   *
   * @param [in]  data  停止閾値
   * @retval	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER setStopLowBright(USHORT data);

  /**
   * @brief 低色温度側引き込み上限座標を設定する
   */
  void setLTempAdjustPnt();

  /**
   * @brief 高色温度側引き込み座標を設定する
   */
  void setHTempAdjustPnt();

  /**
   * @brief Mg側およびG側の引き込み範囲調整を行う
   */
  void setMgGDirAdjust();

  /**
   * @brief 調整後引き込み座標を計算する
   */
  ::ColPoint calcMgGDirAdjust(AWB_Adjust_Dir dir, ::ColPoint stdPnt, ::ColPoint Pnt);

  AXMIspAwbSettingsOnePush OnePush;			// [AWB-5]全引込み実行設定値
  AXMIspAwbSettingsLightRange LightRange;			// [AWB-6]光源枠引込み範囲設定値
  AXMIspAwbSettingsOffset Offset;				// [AWB-7]WBオフセット調整設定値
  AXMIspAwbSettingsAdjustMatrix AdjustMatrix;		// [AWB-8]カラーマトリックス調整設定値
  AXMIspAwbSettingsWB WB;					// [AWB-9]マニュアルモード時ゲイン指定
  AXMIspAwbSettingsConvergence Convergence;		// [AWB-10]AWB遷移特性設定
  AXMIspAwbSettingsLightSourceWeight LightSourceWeight;  // [AWB-11]光源別光源枠重み付け設定値
  AXMIspAwbSettingsOPDSet OPDSet;				// [AWB-13]OPD設定値
  AXMIspAwbSettingsOPDWeight OPDWeight;			// [AWB-14]OPD枠重み付け設定値
  AXMIspAwbSettingsOPDMask OPDMask;			// [AWB-15]OPD枠AWBマスク領域設定
  AXMIspAwbSettingsSpectrumAdjust SpectrumAdjust;		// [AWB-16]センサー分光バラツキ調整
  AXMIspAwbSettingsLightSource LightSource;		// [AWB-17]光源枠詳細調整設定
  AXMIspAwbSettingsLightDefault LightDefault;		// [AWB-18]光源デフォルト位置設定
  AXMIspAwbSettingsOPDTrust OPDTrust;			// [AWB-19]OPD信頼性レベル設定値
  AXMIspAwbSettingsColorMatrix ColorMatrix;		// [Img5.5.2]カラーマトリックス設定値

  AWBMode AWB_Mode;								// [AWB-1]AWBモード選択
  ATWSceaneMode ATW_SceaneMode;							// [AWB-2]ATWシーンモード選択
  USHORT ColorTemp;								// [AWB-3]マニュアルモード時色温度指定
  AWBLightSelect LightSelect;							// [AWB-4]マニュアルモード時照明指定
  USHORT StopLowBright;							// [AWB12]ATW停止閾値

};

#endif /* AXMISPAWBSETTINGS_HPP_ */
