/**
 * @file   AXMIspAwbState.hpp
 * @brief  AXMIspAwbクラス StateMachineオブジェクトヘッダ
 *
 * @n      AWBステートマシン実行処理
 *         Across ISP
 * @par    Programming Language
 *         C++
 */

#define NEW_STATUS

  /**
   * @enum  ispAwbModelState
   * @brief StateMachine用状態定義
   */
  enum ispAwbModelState {
    STATE_WAITING = 0,                  //! 待機中:調整モード(WB調整以外)
    STATE_ATWMode,                      //! ATWモード
    STATE_ManuMode,                     //! マニュアルモード
    STATE_OnePush,                      //! ワンプッシュモード
    STATE_ADJUST,                       //! 調整モード

    STATE_MAX
  };

  /**
   * @enum  ispAwbModelEvent
   * @brief StateMachine用イベント定義
   */
  enum ispAwbModelEvent {
    EVENT_COLOR_DATA = AXFEVENT_MODEL(0),                   //!<  色データ計算処理 開始
    EVENT_OPD_MASK,                                         //!<  OPDマスク処理 開始
    EVENT_LIGHT_SOURCE,                                     //!<  光源枠判定処理 開始
    EVENT_WEIGHT_DATA,                                      //!<  重み付け処理 開始
    EVENT_BALANCE_CENTER,                                   //!<  重心計算処理 開始
    EVENT_TARGET,                                           //!<  調整/収束制御処理 開始
    EVENT_CONTROL,                                          //!<  ゲイン/マトリックス設定処理 開始
    EVENT_TARGET_CONTROL,                                   //!<  重心計算/調整・収束制御/ゲイン・マトリックス設定処理 開始

    EVENT_ATW_MODE,                                         //!<  ATWモード切替
    EVENT_MANU_MODE,                                        //!<  マニュアルモード切替
    EVENT_ONE_PUSH,                                         //!<  ワンプッシュモード切替
    EVENT_ADJUST,                                           //!<  調整モード(WB調整)切替
    EVENT_WAITING,                                          //!<  待機中:調整モード(WB調整以外)切替		// 今回不要　該当処理なし

    EVENT_SPEC_ADDR,                                        //!<  設定値保存先アドレス通知
#ifdef USE_AXFTEST_COMMAND
    EVENT_AXCOMM_TEST = AXFEVENT_MODEL(1000),
    EVENT_PRIORITY_TEST = AXFEVENT_MODEL(1002),
    EVENT_MEM_TEST = AXFEVENT_MODEL(1003),
    EVENT_LOOPBACK_TEST = AXFEVENT_MODEL(1004),
#endif /*USE_AXFTEST_COMMAND*/
  };

  // StateMachine用状態テーブル宣言
  static const AXMIspAwb::StateTable state_wating[];        //!<  調整モード(WB調整以外：待機中) 状態テーブル定義
  static const AXMIspAwb::StateTable state_atw_mode[];      //!<  ATWモード 状態テーブル定義
  static const AXMIspAwb::StateTable state_manu_mode[];     //!<  マニュアルモード 状態テーブル定義
  static const AXMIspAwb::StateTable state_one_push[];      //!<  ワンプッシュモード 状態テーブル定義
  static const AXMIspAwb::StateTable state_adjust[];        //!<  調整モード(WB調整) 状態テーブル定義

  // StateMachine用メンバ関数宣言
  /**
   * @brief 設定値保存先アドレス通知を受信し、設定値参照先にそのアドレスを設定する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcSpecAddr(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整以外：待機中)からATWモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0008(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整以外：待機中)からマニュアルモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0009(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整以外：待機中)からワンプッシュモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix000A(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整以外：待機中)待機中から調整モード(WB調整)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix000B(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整以外：待機中)から調整モード(WB調整以外：待機中)に切り替える(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix000C(const void *pParam, int size);
  /**
   * @brief 色データ計算処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0100(const void *pParam, int size);
  /**
   * @brief OPDマスク処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0101(const void *pParam, int size);
  /**
   * @brief 光源枠判定処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0102(const void *pParam, int size);
  /**
   * @brief 重み付け処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0103(const void *pParam, int size);
  /**
   * @brief 重心計算処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0104(const void *pParam, int size);
  /**
   * @brief 調整・収束制御処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0105(const void *pParam, int size);
  /**
   * @brief ゲイン・マトリックス設定処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0106(const void *pParam, int size);
  /**
   * @brief 重心計算/調整・収束制御/ゲイン・マトリックス設定処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0107(const void *pParam, int size);
  /**
   * @brief ATWモードからATWモードに切り替える(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0108(const void *pParam, int size);
  /**
   * @brief ATWモードからマニュアルモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0109(const void *pParam, int size);
  /**
   * @brief ATWモードからワンプッシュモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix010A(const void *pParam, int size);
  /**
   * @brief ATWモードから調整モード(WB調整)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix010B(const void *pParam, int size);
  /**
   * @brief ATWモードから調整モード(WB調整以外：待機中)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix010C(const void *pParam, int size);
  /**
   * @brief 色データ計算処理を実行する(マニュアルモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0200(const void *pParam, int size);
  /**
   * @brief ゲイン・マトリックス設定処理を実行する(ATWモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0201(const void *pParam, int size);
  /**
   * @brief マニュアルモードからATWモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0208(const void *pParam, int size);
  /**
   * @brief マニュアルモードからマニュアルモードに切り替える(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0209(const void *pParam, int size);
  /**
   * @brief マニュアルモードからワンプッシュモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix020A(const void *pParam, int size);
  /**
   * @brief マニュアルモードから調整モード(WB調整)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix020B(const void *pParam, int size);
  /**
   * @brief マニュアルモードから調整モード(WB調整以外：待機中)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix020C(const void *pParam, int size);
  //  ワンプッシュモード
  /**
   * @brief 色データ計算処理を実行する(ワンプッシュモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0300(const void *pParam, int size);
  /**
   * @brief 光源枠判定処理を実行する(ワンプッシュモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0301(const void *pParam, int size);
  /**
   * @brief 重心計算処理を実行する(ワンプッシュモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0302(const void *pParam, int size);
  /**
   * @brief 調整・収束制御処理を実行する(ワンプッシュモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0303(const void *pParam, int size);
  /**
   * @brief ゲイン・マトリックス処理を実行する(ワンプッシュモード)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0304(const void *pParam, int size);
  /**
   * @brief ワンプッシュモードからATWモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0308(const void *pParam, int size);
  /**
   * @brief ワンプッシュモードからマニュアルモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0309(const void *pParam, int size);
  /**
   * @brief ワンプッシュモードからワンプッシュモードに切り替える(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix030A(const void *pParam, int size);
  /**
   * @brief ワンプッシュモードから調整モード(WB調整)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix030B(const void *pParam, int size);
  /**
   * @brief ワンプッシュモードから調整モード(WB調整以外：待機中)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix030C(const void *pParam, int size);
  /**
   * @brief バラつき調整を実行する(調整モード：WB調整)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1000(const void *pParam, int size);                 //  バラつき調整
  /**
   * @brief 調整モード(WB調整)からATWモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1008(const void *pParam, int size);                 //  ATWモード切替
  /**
   * @brief 調整モード(WB調整)からマニュアルモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix1009(const void *pParam, int size);                 //  マニュアルモード切替
  /**
   * @brief 調整モード(WB調整)からワンプッシュモードに切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix100A(const void *pParam, int size);                 //  ワンプッシュモード切替
  /**
   * @brief 調整モード(WB調整)から調整モード(WB調整)に切り替える(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix100B(const void *pParam, int size);
  /**
   * @brief 調整モード(WB調整)から調整モード(WB調整以外：待機中)に切り替える
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix100C(const void *pParam, int size);

  /**
   * @brief 設定値の監視処理を実行する
   *
   * @retval 	AWB_E_OK	正常終了
   * @retval	AWB_E_ER	異常終了
   */
  AWB_ER	CheckParameter();
  /**
   * @brief 判定用光源枠の更新処理
   *
   * @n
   */
  void refleshJudgedLightSourceData();

#ifdef USE_AXFTEST_COMMAND
  ax::actorFuncStatus funcAxcommTest(const void *pParam, int size);
  ax::actorFuncStatus funcPriorityTest(const void *pParam, int size);
  ax::actorFuncStatus funcMemTest(const void *pParam, int size);
  ax::actorFuncStatus funcLoopbackTest(const void *pParam, int size);
#endif /*USE_AXFTEST_COMMAND*/

  int   state;                                              //! 次期ステータス格納用(setNextStateの引数)
