/**
 * @file   AXMIspCorrectState.hpp
 * @brief  AXMIspCorrectクラス StateMachineオブジェクトヘッダ
 *
 * @n
 *         Across ISP
 * @par    Programming Language
 *         C++
 */

#define NEW_STATUS

  /**
   * @enum  ispCorrectModelState
   * @brief StateMachine用状態定義
   */
  enum ispCorrectModelState {
    STATE_WAITING = 0,                  //!<  待機中
    STATE_RUNNING,                      //!<  シェーディング補正モード
    STATE_KIZU_RUNNING,                 //!<  キズ補正モード

    STATE_MAX
  };

  /**
   * @enum  ispCorrectModelEvent
   * @brief StateMachine用イベント定義
   */
  enum ispCorrectModelEvent {
    EVENT_WAITING = AXFEVENT_MODEL(0),  //!<  待機中移行通知
    EVENT_RUNNING,                      //!<  シェーディング補正モード移行通知
    EVENT_KIZU_RUNNING,                 //!<  キズ補正モード移行通知
    EVENT_START,                        //!<  演算開始
    EVENT_KIZU_START,                   //!<  キズ補正演算開始

    EVENT_SPEC_ADDR,                    //!<  設定値保存先アドレス通知
#ifdef USE_AXFTEST_COMMAND
    EVENT_AXCOMM_TEST = AXFEVENT_MODEL(1000),
    EVENT_PRIORITY_TEST = AXFEVENT_MODEL(1002),
    EVENT_MEM_TEST = AXFEVENT_MODEL(1003),
    EVENT_LOOPBACK_TEST = AXFEVENT_MODEL(1004),
#endif /*USE_AXFTEST_COMMAND*/
  };

  // StateMachine用状態テーブル宣言
  static const AXMIspCorrect::StateTable state_waiting[];                       //!<  待機中 状態テーブル定義
  static const AXMIspCorrect::StateTable state_running[];                       //!<  シェーディング補正モード 状態テーブル定義
  static const AXMIspCorrect::StateTable state_kizu_running[];                  //!<  キズ補正モード 状態テーブル定義

  // StateMachine用メンバ関数宣言
  /**
   * @brief 設定値保存先アドレス通知を受信し、設定値参照先にそのアドレスを設定する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus funcSpecAddr(const void *pParam, int size);
  /**
   * @brief 待機中からシェーディング補正モードに移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0008(const void *pParam, int size);
  /**
   * @brief 待機中から待機中に移行(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0009(const void *pParam, int size);
  /**
   * @brief 待機中からキズ補正モードに移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix000A(const void *pParam, int size);
  /**
   * @brief シェーディング補正演算処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0100(const void *pParam, int size);
  /**
   * @brief シェーディング補正モードからシェーディング補正モードに移行(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0108(const void *pParam, int size);
  /**
   * @brief シェーディング補正モードから待機中に移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0109(const void *pParam, int size);
  /**
   * @brief シェーディング補正モードからキズ補正モードに移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix010A(const void *pParam, int size);
  /**
   * @brief キズ補正演算処理を実行する
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0200(const void *pParam, int size);
  /**
   * @brief キズ補正モードからシェーディング補正モードに移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0208(const void *pParam, int size);
  /**
   * @brief キズ補正モードから待機中に移行
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix0209(const void *pParam, int size);
  /**
   * @brief キズ補正モードからキズ補正モードに移行(dummy)
   *
   * @param [in] pParam 受信パラメータ
   * @param [in] size 受信パラメータサイズ
   * @return     実行完了状態(正常終了)
   */
  ax::actorFuncStatus matrix020A(const void *pParam, int size);

#ifdef USE_AXFTEST_COMMAND
  ax::actorFuncStatus funcAxcommTest(const void *pParam, int size);
  ax::actorFuncStatus funcPriorityTest(const void *pParam, int size);
  ax::actorFuncStatus funcMemTest(const void *pParam, int size);
  ax::actorFuncStatus funcLoopbackTest(const void *pParam, int size);
#endif /*USE_AXFTEST_COMMAND*/

  /**
   * @enum Shade_IntMode
   * @n シェーディング補正 積算モード
   */
  typedef enum {
    Shade_IntMode_Set = 0,                                  //!<  積算データセット
    Shade_IntMode_Add,                                      //!<  積算データ加算
  } Shade_IntMode;

  /**
   * @brief シェーディング補正 補正データ演算用積算データ作成
   *
   * @param [in] p_img 画像データ格納先アドレス
   * @param [in] mode 積算モード
   */
  void funcIntData(UCHAR *p_img, Shade_IntMode mode);

  /**
   * @brief シェーディング補正 補正データ演算処理
   */
  void funcCorrectData();

  /**
   * @brief 画素欠陥補正 画素検出処理
   *
   * @param [in] p_Img 画像データ格納先アドレス
   */
  void funcDefectData(USHORT *p_Img);

  /**
   * @brief 画素欠陥補正 輝度レベル順に並び替え
   */
  void funcSortDefectData_PriorityBright();

  /**
   * @brief 画素欠陥補正 連欠陥順に並び替え
   */
  void funcSortDefectData_PrioritySerial();

  /**
   * @brief 画素欠陥補正 アドレスデータ初期化
   *
   * @return 最大登録データ数
   */
  INT initDefectData();

  /**
   * @brief  画素欠陥補正 アドレスデータセット
   */
  void setDefectData();

  /**
   * @brief  画素欠陥補正 内部アドレスデータセット
   */
  void setDefectData_Internal();

  /**
   * @brief 画素欠陥補正 外部アドレスデータセット
   */
  void setDefectData_External();
